"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "refresh", {
    enumerable: true,
    get: function() {
        return refresh;
    }
});
const _config = /*#__PURE__*/ _interop_require_default(require("config"));
const _expressasynchandler = /*#__PURE__*/ _interop_require_default(require("express-async-handler"));
const _bcrypt = /*#__PURE__*/ _interop_require_default(require("bcrypt"));
const _jsonwebtoken = /*#__PURE__*/ _interop_require_default(require("jsonwebtoken"));
const _constants = require("@tamanu/constants");
const _errors = require("@tamanu/shared/errors");
const _utils = require("./utils");
function _interop_require_default(obj) {
    return obj && obj.__esModule ? obj : {
        default: obj
    };
}
const refresh = ({ secret, refreshSecret })=>(0, _expressasynchandler.default)(async (req, res)=>{
        const { body, store } = req;
        const { refreshToken, deviceId } = body;
        const { canonicalHostName, auth } = _config.default;
        const { tokenDuration, saltRounds, refreshToken: { refreshIdLength, tokenDuration: refreshTokenDuration, absoluteExpiration } } = auth;
        if (!(0, _utils.isInternalClient)(req.header('X-Tamanu-Client'))) {
            throw new _errors.BadAuthenticationError('Invalid client');
        }
        let contents = null;
        try {
            contents = await (0, _utils.verifyToken)(refreshToken, refreshSecret, {
                audience: _constants.JWT_TOKEN_TYPES.REFRESH,
                issuer: canonicalHostName
            });
        } catch (e) {
            throw new _errors.BadAuthenticationError('Invalid token (jMbP)');
        }
        const { userId, refreshId } = contents;
        const user = await store.models.User.findOne({
            where: {
                id: userId,
                visibilityStatus: _constants.VISIBILITY_STATUSES.CURRENT
            }
        });
        if (!user) {
            throw new _errors.BadAuthenticationError('Invalid token (vN3y)');
        }
        const dbEntry = await store.models.RefreshToken.findOne({
            where: {
                userId,
                deviceId
            }
        });
        if (!dbEntry) {
            throw new _errors.BadAuthenticationError('Invalid token (J7GC)');
        }
        if (dbEntry.expiresAt < new Date()) {
            throw new _errors.BadAuthenticationError('Refresh token expired');
        }
        const refreshIdValid = await _bcrypt.default.compare(refreshId, dbEntry.refreshId);
        if (!refreshIdValid) {
            throw new _errors.BadAuthenticationError('Invalid token (Xh01)');
        }
        // issue new access token
        const accessTokenJwtId = (0, _utils.getRandomU32)();
        const token = await (0, _utils.buildToken)({
            userId: user.id,
            deviceId
        }, secret, {
            expiresIn: tokenDuration,
            audience: _constants.JWT_TOKEN_TYPES.ACCESS,
            issuer: canonicalHostName,
            jwtid: `${accessTokenJwtId}`
        });
        // rotate refresh token
        const newRefreshId = await (0, _utils.getRandomBase64String)(refreshIdLength);
        const refreshTokenJwtId = (0, _utils.getRandomU32)();
        const hashedRefreshId = await _bcrypt.default.hash(newRefreshId, saltRounds);
        const newRefreshToken = await (0, _utils.buildToken)({
            userId: user.id,
            refreshId: newRefreshId,
            // If absolute expiration pass through the exp from the old token
            ...absoluteExpiration && {
                exp: contents.exp
            }
        }, refreshSecret, {
            audience: _constants.JWT_TOKEN_TYPES.REFRESH,
            issuer: canonicalHostName,
            jwtid: `${refreshTokenJwtId}`,
            ...!absoluteExpiration && {
                expiresIn: refreshTokenDuration
            }
        });
        // Extract expiry as set by jwt.sign
        const { exp } = _jsonwebtoken.default.decode(newRefreshToken);
        await store.models.RefreshToken.upsert({
            refreshId: hashedRefreshId,
            userId: user.id,
            deviceId,
            expiresAt: absoluteExpiration ? dbEntry.expiresAt : new Date(exp * 1000)
        }, {
            where: {
                userId: user.id,
                deviceId
            }
        });
        res.send({
            token,
            refreshToken: newRefreshToken
        });
    });

//# sourceMappingURL=refresh.js.map