"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "surveyResponse", {
    enumerable: true,
    get: function() {
        return surveyResponse;
    }
});
const _express = /*#__PURE__*/ _interop_require_default(require("express"));
const _expressasynchandler = /*#__PURE__*/ _interop_require_default(require("express-async-handler"));
const _ability = require("@casl/ability");
const _transformAnswers = require("@tamanu/shared/reports/utilities/transformAnswers");
const _constants = require("@tamanu/constants");
const _errors = require("@tamanu/shared/errors");
function _interop_require_default(obj) {
    return obj && obj.__esModule ? obj : {
        default: obj
    };
}
const surveyResponse = _express.default.Router();
surveyResponse.get('/:id', (0, _expressasynchandler.default)(async (req, res)=>{
    const { models, params, query } = req;
    req.checkPermission('read', 'SurveyResponse');
    const surveyResponseRecord = await models.SurveyResponse.findByPk(params.id);
    const survey = await surveyResponseRecord.getSurvey();
    req.checkPermission('read', survey);
    const components = await models.SurveyScreenComponent.getComponentsForSurvey(surveyResponseRecord.surveyId, {
        includeAllVitals: true
    });
    const answers = await models.SurveyResponseAnswer.findAll({
        where: {
            responseId: params.id
        }
    });
    const transformedAnswers = await (0, _transformAnswers.transformAnswers)(models, answers, components, {
        notTransformDate: true
    });
    await req.audit.access({
        recordId: surveyResponseRecord.id,
        params,
        model: models.SurveyResponse,
        facilityId: query.facilityId
    });
    res.send({
        ...surveyResponseRecord.forResponse(),
        components,
        answers: answers.map((answer)=>{
            const transformedAnswer = transformedAnswers.find((a)=>a.id === answer.id);
            return {
                ...answer.dataValues,
                originalBody: answer.body,
                body: transformedAnswer?.body,
                sourceType: transformedAnswer?.sourceType
            };
        })
    });
}));
surveyResponse.post('/$', (0, _expressasynchandler.default)(async (req, res)=>{
    const { models, body: { facilityId, ...body }, db, settings } = req;
    // Responses for the vitals survey will check against 'Vitals' create permissions
    // All others will check against 'SurveyResponse' create permissions
    const noun = await models.Survey.getResponsePermissionCheck(body.surveyId);
    if (noun === 'Charting') {
        req.checkPermission('create', (0, _ability.subject)('Charting', {
            id: body.surveyId
        }));
    } else {
        req.checkPermission('create', noun);
    }
    const getDefaultId = async (type)=>models.SurveyResponseAnswer.getDefaultId(type, settings[facilityId]);
    const updatedBody = {
        locationId: body.locationId || await getDefaultId('location'),
        departmentId: body.departmentId || await getDefaultId('department'),
        userId: req.user.id,
        facilityId,
        ...body
    };
    const responseRecord = await db.transaction(async ()=>{
        return models.SurveyResponse.createWithAnswers(updatedBody);
    });
    res.send(responseRecord);
}));
surveyResponse.put('/complexChartInstance/:id', (0, _expressasynchandler.default)(async (req, res)=>{
    const { models, body, params, db } = req;
    const responseRecord = await models.SurveyResponse.findByPk(params.id);
    if (!responseRecord) {
        throw new _errors.NotFoundError('Response record not found');
    }
    req.checkPermission('write', (0, _ability.subject)('Charting', {
        id: responseRecord.surveyId
    }));
    const survey = await responseRecord.getSurvey();
    if (survey.surveyType !== _constants.SURVEY_TYPES.COMPLEX_CHART_CORE) {
        throw new _errors.InvalidOperationError('Cannot edit survey responses');
    }
    const components = await models.SurveyScreenComponent.getComponentsForSurvey(survey.id);
    const responseAnswers = await models.SurveyResponseAnswer.findAll({
        where: {
            responseId: params.id
        }
    });
    await db.transaction(async ()=>{
        for (const [dataElementId, value] of Object.entries(body.answers)){
            if (!components.some((c)=>c.dataElementId === dataElementId)) {
                throw new _errors.InvalidOperationError('Some components are missing from the survey');
            }
            // Ignore null values
            if (value === null) {
                continue;
            }
            const existingAnswer = responseAnswers.find((a)=>a.dataElementId === dataElementId);
            if (existingAnswer) {
                await existingAnswer.update({
                    body: value
                });
            } else {
                await models.SurveyResponseAnswer.create({
                    dataElementId,
                    body: value,
                    responseId: params.id
                });
            }
        }
    });
    res.send(responseRecord);
}));

//# sourceMappingURL=surveyResponse.js.map