"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "EncounterPausePrescription", {
    enumerable: true,
    get: function() {
        return EncounterPausePrescription;
    }
});
const _constants = require("@tamanu/constants");
const _sequelize = require("sequelize");
const _Model = require("./Model");
const _buildEncounterLinkedSyncFilter = require("../sync/buildEncounterLinkedSyncFilter");
const _dateTime = require("@tamanu/utils/dateTime");
const _model = require("../types/model");
const _buildEncounterLinkedLookupFilter = require("../sync/buildEncounterLinkedLookupFilter");
let EncounterPausePrescription = class EncounterPausePrescription extends _Model.Model {
    static initModel({ primaryKey, ...options }) {
        super.init({
            id: primaryKey,
            pauseDuration: {
                type: _sequelize.DataTypes.DECIMAL,
                allowNull: false
            },
            pauseTimeUnit: {
                type: _sequelize.DataTypes.STRING,
                allowNull: false
            },
            pauseStartDate: (0, _model.dateTimeType)('pauseStartDate', {
                allowNull: false,
                defaultValue: _dateTime.getCurrentDateTimeString
            }),
            pauseEndDate: (0, _model.dateTimeType)('pauseEndDate', {
                allowNull: false
            }),
            notes: {
                type: _sequelize.DataTypes.TEXT,
                allowNull: true
            }
        }, {
            ...options,
            tableName: 'encounter_pause_prescriptions',
            syncDirection: _constants.SYNC_DIRECTIONS.BIDIRECTIONAL,
            hooks: {
                // Create history records
                afterCreate: async (instance, options)=>{
                    // Get the EncounterPausePrescriptionHistory model
                    const historyModel = instance.sequelize.models.EncounterPausePrescriptionHistory;
                    if (!historyModel) return;
                    // Create a history record for the pause action
                    await historyModel.create({
                        encounterPrescriptionId: instance.encounterPrescriptionId,
                        action: 'pause',
                        actionDate: instance.pauseStartDate,
                        actionUserId: instance.pausingClinicianId,
                        notes: instance.notes,
                        pauseDuration: instance.pauseDuration,
                        pauseTimeUnit: instance.pauseTimeUnit
                    }, {
                        transaction: options.transaction
                    });
                },
                afterUpdate: async (instance, options)=>{
                    const historyModel = instance.sequelize.models.EncounterPausePrescriptionHistory;
                    if (!historyModel) return;
                    const currentDate = (0, _dateTime.getCurrentDateTimeString)();
                    // Check if this is a manual resume (pauseEndDate was changed to be <= current date)
                    if (instance.pauseEndDate && new Date(instance.pauseEndDate).getTime() <= new Date(currentDate).getTime()) {
                        // Create a history record for the resume action
                        await historyModel.create({
                            encounterPrescriptionId: instance.encounterPrescriptionId,
                            action: 'resume',
                            actionDate: instance.pauseEndDate,
                            actionUserId: instance.updatedBy,
                            notes: instance.notes
                        }, {
                            transaction: options.transaction
                        });
                    }
                }
            }
        });
    }
    static initRelations(models) {
        this.belongsTo(models.EncounterPrescription, {
            foreignKey: 'encounterPrescriptionId',
            as: 'encounterPrescription'
        });
        this.belongsTo(models.User, {
            foreignKey: 'pausingClinicianId',
            as: 'pausingClinician'
        });
        this.belongsTo(models.User, {
            foreignKey: 'createdBy',
            as: 'createdByUser'
        });
        this.belongsTo(models.User, {
            foreignKey: 'updatedBy',
            as: 'updatedByUser'
        });
    }
    static buildPatientSyncFilter(patientCount, markedForSyncPatientsTable) {
        if (patientCount === 0) {
            return null;
        }
        return (0, _buildEncounterLinkedSyncFilter.buildEncounterLinkedSyncFilter)([
            this.tableName,
            'encounter_prescriptions',
            'encounters'
        ], markedForSyncPatientsTable);
    }
    static buildSyncLookupQueryDetails() {
        return {
            select: (0, _buildEncounterLinkedLookupFilter.buildEncounterLinkedLookupSelect)(this),
            joins: (0, _buildEncounterLinkedLookupFilter.buildEncounterLinkedLookupJoins)(this, [
                'encounter_prescriptions',
                'encounters'
            ])
        };
    }
    /**
   * Check if a prescription is currently paused
   * @param {string} prescriptionId - The ID of the prescription to check
   * @returns {Promise<{isPaused: boolean, pauseData?: EncounterPausePrescription}>} Whether the medication is paused and the pause data if it is
   */ static async isPrescriptionPaused(prescriptionId) {
        try {
            // Get the models from sequelize
            const models = this.sequelize.models;
            const EncounterPrescription = models.EncounterPrescription;
            // Find the encounter prescription link (assuming 1:1 relationship)
            const encounterPrescription = await EncounterPrescription.findOne({
                where: {
                    prescriptionId
                }
            });
            if (!encounterPrescription) {
                return {
                    isPaused: false
                };
            }
            // Check if there's an active pause record for this encounter-prescription
            const currentDate = (0, _dateTime.getCurrentDateTimeString)();
            const activePause = await this.findOne({
                where: {
                    encounterPrescriptionId: encounterPrescription.id,
                    pauseEndDate: {
                        [_sequelize.Op.gt]: currentDate
                    }
                },
                order: [
                    [
                        'createdAt',
                        'DESC'
                    ]
                ]
            });
            return {
                isPaused: !!activePause,
                pauseData: activePause || undefined
            };
        } catch (error) {
            console.error('Error checking if prescription is paused:', error);
            return {
                isPaused: false
            };
        }
    }
};

//# sourceMappingURL=EncounterPausePrescription.js.map