function _define_property(obj, key, value) {
    if (key in obj) {
        Object.defineProperty(obj, key, {
            value: value,
            enumerable: true,
            configurable: true,
            writable: true
        });
    } else {
        obj[key] = value;
    }
    return obj;
}
import { SYNC_DIRECTIONS, DEFAULT_LANGUAGE_CODE } from '@tamanu/constants';
import { DataTypes, Op } from 'sequelize';
import { Model } from './Model';
import { keyBy, mapValues } from 'lodash';
import { translationFactory } from '@tamanu/shared/utils/translation/translationFactory';
import { getEnumPrefix } from '@tamanu/shared/utils/enumRegistry';
export class TranslatedString extends Model {
    static initModel(options) {
        super.init({
            id: {
                // translated_string records use a generated primary key that enforces one per string and language,
                type: `TEXT GENERATED ALWAYS AS ("string_id" || ';' || "language") STORED`,
                set () {
                // any sets of the convenience generated "id" field can be ignored
                }
            },
            stringId: {
                type: DataTypes.TEXT,
                allowNull: false,
                primaryKey: true,
                validate: {
                    doesNotContainIdDelimiter: (value)=>{
                        if (value.includes(';')) {
                            throw new Error('Translation ID cannot contain ";"');
                        }
                    }
                }
            },
            language: {
                type: DataTypes.TEXT,
                allowNull: false,
                primaryKey: true,
                validate: {
                    doesNotContainIdDelimiter: (value)=>{
                        if (value.includes(';')) {
                            throw new Error('Language cannot contain ";"');
                        }
                    }
                }
            },
            text: {
                type: DataTypes.TEXT,
                allowNull: false
            }
        }, {
            ...options,
            syncDirection: SYNC_DIRECTIONS.BIDIRECTIONAL,
            indexes: [
                {
                    name: 'string_language_unique',
                    fields: [
                        'string_id',
                        'language'
                    ],
                    unique: true
                },
                {
                    name: 'string_id_index',
                    fields: [
                        'string_id'
                    ]
                },
                {
                    name: 'language_index',
                    fields: [
                        'language'
                    ]
                },
                {
                    name: 'updated_at_sync_tick_index',
                    fields: [
                        'language',
                        'updated_at_sync_tick'
                    ]
                }
            ]
        });
    }
    static buildSyncFilter() {
        return null; // syncs everywhere
    }
    static buildSyncLookupQueryDetails() {
        return null; // syncs everywhere
    }
}
_define_property(TranslatedString, "getPossibleLanguages", async ()=>{
    const languagesInDb = await TranslatedString.findAll({
        attributes: [
            'language'
        ],
        group: 'language',
        where: {
            language: {
                [Op.not]: DEFAULT_LANGUAGE_CODE
            }
        }
    });
    const languageNames = await TranslatedString.findAll({
        where: {
            stringId: 'languageName'
        }
    });
    const countryCodes = await TranslatedString.findAll({
        where: {
            stringId: 'countryCode'
        }
    });
    return {
        languagesInDb,
        languageNames,
        countryCodes
    };
});
_define_property(TranslatedString, "getTranslations", async (language, prefixIds)=>{
    const translatedStringRecords = await TranslatedString.findAll({
        where: {
            language,
            // filter Boolean to avoid query all records
            [Op.or]: prefixIds.filter(Boolean).map((prefixId)=>({
                    id: {
                        [Op.startsWith]: prefixId
                    }
                }))
        },
        attributes: [
            'stringId',
            'text'
        ]
    });
    const translations = mapValues(keyBy(translatedStringRecords, 'stringId'), 'text');
    return translations;
});
_define_property(TranslatedString, "getTranslationFunction", async (language, prefixIds = [])=>{
    const translations = await TranslatedString.getTranslations(language, prefixIds);
    return (stringId, fallback, translationOptions = {})=>{
        const translationFunc = translationFactory(translations);
        const { value } = translationFunc(stringId, fallback, translationOptions);
        return value;
    };
});
_define_property(TranslatedString, "getEnumTranslationFunction", async (language, prefixIds = [])=>{
    const translationFunc = await TranslatedString.getTranslationFunction(language, prefixIds);
    return (enumValues, currentValue)=>{
        const fallback = enumValues[currentValue];
        const stringId = `${getEnumPrefix(enumValues)}.${currentValue}`;
        const value = translationFunc(stringId, fallback || '');
        return value;
    };
});

//# sourceMappingURL=TranslatedString.js.map