function _define_property(obj, key, value) {
    if (key in obj) {
        Object.defineProperty(obj, key, {
            value: value,
            enumerable: true,
            configurable: true,
            writable: true
        });
    } else {
        obj[key] = value;
    }
    return obj;
}
import { DataTypes } from 'sequelize';
import { SYNC_DIRECTIONS, VISIBILITY_STATUSES } from '@tamanu/constants';
import { InvalidOperationError } from '@tamanu/shared/errors';
import { Model } from './Model';
import { buildPatientSyncFilterViaPatientId } from '../sync/buildPatientSyncFilterViaPatientId';
import { buildPatientLinkedLookupFilter } from '../sync/buildPatientLinkedLookupFilter';
import { dateType } from '../types/model';
export class PatientDeathData extends Model {
    static initModel({ primaryKey, ...options }) {
        super.init({
            id: primaryKey,
            birthWeight: {
                type: DataTypes.INTEGER,
                unsigned: true
            },
            carrierAge: {
                type: DataTypes.INTEGER,
                unsigned: true
            },
            carrierPregnancyWeeks: {
                type: DataTypes.INTEGER,
                unsigned: true
            },
            externalCauseDate: dateType('externalCauseDate'),
            lastSurgeryDate: dateType('lastSurgeryDate'),
            externalCauseLocation: DataTypes.STRING,
            externalCauseNotes: DataTypes.TEXT,
            fetalOrInfant: DataTypes.BOOLEAN,
            hoursSurvivedSinceBirth: {
                type: DataTypes.INTEGER,
                unsigned: true
            },
            manner: DataTypes.STRING,
            pregnancyContributed: DataTypes.STRING,
            recentSurgery: DataTypes.STRING,
            stillborn: DataTypes.STRING,
            wasPregnant: DataTypes.STRING,
            withinDayOfBirth: DataTypes.BOOLEAN,
            outsideHealthFacility: DataTypes.BOOLEAN,
            primaryCauseTimeAfterOnset: DataTypes.INTEGER,
            antecedentCause1TimeAfterOnset: DataTypes.INTEGER,
            antecedentCause2TimeAfterOnset: DataTypes.INTEGER,
            antecedentCause3TimeAfterOnset: DataTypes.INTEGER,
            isFinal: DataTypes.BOOLEAN,
            visibilityStatus: {
                type: DataTypes.TEXT,
                defaultValue: VISIBILITY_STATUSES.CURRENT
            }
        }, {
            ...options,
            syncDirection: SYNC_DIRECTIONS.BIDIRECTIONAL,
            tableName: 'patient_death_data',
            validate: {
                mustHavePatient () {
                    if (this.deletedAt) return;
                    if (!this.patientId) {
                        throw new InvalidOperationError('Patient death data must have a patient.');
                    }
                },
                mustHaveClinician () {
                    if (this.deletedAt) return;
                    if (!this.clinicianId) {
                        throw new InvalidOperationError('Patient death data must have a clinician.');
                    }
                },
                yesNoUnknownFields () {
                    if (this.deletedAt) return;
                    for (const field of [
                        'recentSurgery',
                        'wasPregnant',
                        'pregnancyContributed',
                        'stillborn'
                    ]){
                        if (this[field] && ![
                            'yes',
                            'no',
                            'unknown'
                        ].includes(this[field])) {
                            throw new InvalidOperationError(`${field} must be 'yes', 'no', 'unknown', or null`);
                        }
                    }
                }
            }
        });
    }
    static initRelations(models) {
        this.belongsTo(models.Patient, {
            foreignKey: 'patientId'
        });
        this.belongsTo(models.User, {
            foreignKey: 'clinicianId',
            as: 'clinician'
        });
        this.belongsTo(models.Facility, {
            foreignKey: 'facilityId',
            as: 'facility'
        });
        // conceptually "hasOne" but we want the foreign key to be here
        this.belongsTo(models.ReferenceData, {
            foreignKey: 'primaryCauseConditionId',
            as: 'primaryCauseCondition'
        });
        this.belongsTo(models.ReferenceData, {
            foreignKey: 'antecedentCause1ConditionId',
            as: 'antecedentCause1Condition'
        });
        this.belongsTo(models.ReferenceData, {
            foreignKey: 'antecedentCause2ConditionId',
            as: 'antecedentCause2Condition'
        });
        this.belongsTo(models.ReferenceData, {
            foreignKey: 'antecedentCause3ConditionId',
            as: 'antecedentCause3Condition'
        });
        this.belongsTo(models.ReferenceData, {
            foreignKey: 'lastSurgeryReasonId',
            as: 'lastSurgeryReason'
        });
        this.belongsTo(models.ReferenceData, {
            foreignKey: 'carrierExistingConditionId',
            as: 'carrierExistingCondition'
        });
        this.hasMany(models.ContributingDeathCause, {
            foreignKey: 'patientDeathDataId',
            as: 'contributingCauses'
        });
    }
    static buildSyncLookupQueryDetails() {
        return buildPatientLinkedLookupFilter(this);
    }
}
_define_property(PatientDeathData, "buildPatientSyncFilter", buildPatientSyncFilterViaPatientId);

//# sourceMappingURL=PatientDeathData.js.map