"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "routes", {
    enumerable: true,
    get: function() {
        return routes;
    }
});
const _sequelize = require("sequelize");
const _express = /*#__PURE__*/ _interop_require_default(require("express"));
const _expressasynchandler = /*#__PURE__*/ _interop_require_default(require("express-async-handler"));
const _config = /*#__PURE__*/ _interop_require_default(require("config"));
const _lodash = require("lodash");
const _datefns = require("date-fns");
const _datefnstz = require("date-fns-tz");
const _fhir = require("@tamanu/constants/fhir");
const _datetime = require("@tamanu/shared/utils/fhir/datetime");
const _requireClientHeaders = require("../../middleware/requireClientHeaders");
const _errors = require("@tamanu/shared/errors");
function _interop_require_default(obj) {
    return obj && obj.__esModule ? obj : {
        default: obj
    };
}
const routes = _express.default.Router();
const COUNTRY_TIMEZONE = _config.default?.countryTimeZone;
// Workaround for this test changing from a hotfix, see EPI-483/484
function formatDate(date) {
    if (!date) return date;
    return (0, _datefnstz.formatInTimeZone)((0, _datefns.parseISO)((0, _datetime.formatFhirDate)(date, _fhir.FHIR_DATETIME_PRECISION.SECONDS_WITH_TIMEZONE)), '+00:00', "yyyy-MM-dd'T'HH:mm:ssXXX").replace(/Z$/, '+00:00');
}
const reportQuery = `
SELECT 
  last_updated::timestamptz at time zone 'UTC' as last_updated,
  patient_id,
  first_name,
  last_name,
  date_of_birth,
  age,
  sex,
  patient_billing_type,
  encounter_id,
  encounter_start_date,
  encounter_end_date,
  discharge_date,
  encounter_type,
  weight,
  visit_type,
  episode_end_status,
  encounter_discharge_disposition,
  triage_category,
  wait_time,
  departments,
  locations,
  reason_for_encounter,
  diagnoses,
  medications,
  vaccinations,
  procedures,
  lab_requests,
  imaging_requests,
  notes
FROM fhir.non_fhir_medici_report

WHERE true
  AND coalesce(patient_billing_id, '-') LIKE coalesce($billing_type, '%%')
  AND encounter_end_date IS NOT NULL
  AND CASE WHEN coalesce($from_date, 'not_a_date') != 'not_a_date'
    THEN last_updated >= $from_date::timestamptz at time zone $timezone_string
  ELSE
    true
  END
  AND CASE WHEN coalesce($to_date, 'not_a_date') != 'not_a_date'
    THEN last_updated <= $to_date::timestamptz at time zone $timezone_string
  ELSE
    true
  END
  AND CASE WHEN coalesce(array_length($input_encounter_ids::varchar[], 1), 0) != 0
    THEN encounter_id = ANY(SELECT unnest($input_encounter_ids::varchar[]))
  ELSE
    true
  END

ORDER BY last_updated DESC
LIMIT $limit OFFSET $offset;
`;
const parseDateParam = (date)=>{
    const { plain: parsedDate } = (0, _datetime.parseDateTime)(date, {
        withTz: COUNTRY_TIMEZONE
    });
    return parsedDate || null;
};
const checkTimePeriod = (fromDate, toDate)=>{
    const fromParsed = new Date(parseDateParam(fromDate));
    const toParsed = new Date(parseDateParam(toDate));
    // Check if start & end time are within 1 hour
    return Math.abs(toParsed - fromParsed) <= 60 * 60 * 1000;
};
routes.use(_requireClientHeaders.requireClientHeaders);
routes.get('/', (0, _expressasynchandler.default)(async (req, res)=>{
    const { sequelize } = req.store;
    const { 'period.start': fromDate, 'period.end': toDate, limit = 100, encounters, offset = 0 } = req.query;
    if (!COUNTRY_TIMEZONE) {
        throw new Error('A countryTimeZone must be configured in local.json5 for this report to run');
    }
    if (!encounters && (!fromDate || !toDate)) {
        throw new _errors.InvalidOperationError('Must provide either an encounters list or both period.start and period.end query parameters');
    }
    if (fromDate && !toDate) {
        throw new _errors.InvalidOperationError('Must provide a period.end if proving a period.start query parameter');
    }
    if (!fromDate && toDate) {
        throw new _errors.InvalidOperationError('Must provide a period.start if proving a period.end query parameter');
    }
    if (fromDate && toDate && !checkTimePeriod(fromDate, toDate)) {
        throw new _errors.InvalidOperationError('The time period must be within 1 hour');
    }
    const data = await sequelize.query(reportQuery, {
        type: _sequelize.QueryTypes.SELECT,
        bind: {
            from_date: fromDate ? parseDateParam(fromDate, COUNTRY_TIMEZONE) : null,
            to_date: toDate ? parseDateParam(toDate, COUNTRY_TIMEZONE) : null,
            input_encounter_ids: encounters?.split(',') ?? [],
            billing_type: null,
            limit: parseInt(limit, 10),
            offset,
            timezone_string: COUNTRY_TIMEZONE
        }
    });
    const mapNotes = (notes)=>notes?.map((note)=>({
                ...note,
                noteDate: formatDate(note.noteDate)
            }));
    const mappedData = data.map((encounterData)=>{
        const encounter = (0, _lodash.mapKeys)(encounterData, (_v, k)=>(0, _lodash.camelCase)(k));
        return {
            ...encounter,
            weight: parseFloat(encounter.weight),
            encounterStartDate: formatDate(new Date(encounter.encounterStartDate)),
            encounterEndDate: formatDate(new Date(encounter.encounterEndDate)),
            dischargeDate: formatDate(new Date(encounter.dischargeDate)),
            sex: (0, _lodash.upperFirst)(encounter.sex),
            departments: encounter.departments?.map((department)=>({
                    ...department,
                    assignedTime: formatDate(department.assignedTime)
                })),
            locations: encounter.locations?.map((location)=>({
                    ...location,
                    assignedTime: formatDate(location.assignedTime)
                })),
            imagingRequests: encounter.imagingRequests?.map((ir)=>({
                    ...ir,
                    notes: mapNotes(ir.notes)
                })),
            labRequests: encounter.labRequests?.map((lr)=>({
                    ...lr,
                    notes: mapNotes(lr.notes)
                })),
            procedures: encounter.procedures?.map((procedure)=>({
                    ...procedure,
                    date: formatDate(procedure.date)
                })),
            notes: mapNotes(encounter.notes),
            encounterType: encounter.encounterType?.map((encounterType)=>({
                    ...encounterType,
                    startDate: formatDate(encounterType.startDate)
                })),
            hoursOfVentilation: 0,
            leaveDays: 0,
            lastUpdated: formatDate(encounter.lastUpdated),
            medications: encounter.medications?.filter((medication)=>!medication.isSensitive)?.map(({ isSensitive, ...medication })=>medication)
        };
    });
    res.status(200).send({
        data: mappedData
    });
}));

//# sourceMappingURL=routes.js.map