export class Node {
    /**
     * Create a new node with the given name
     *
     * @param {string} name
     * @returns {Node}
     */
    static create(name: string): Node;
    /**
     * @readonly
     */
    static readonly type: "node";
    /**
     * @param {Identifier} name
     * @param {Entry[]} [entries]
     * @param {Document | null} [children]
     */
    constructor(name: Identifier, entries?: Entry[] | undefined, children?: Document | null | undefined);
    /**
     * @readonly
     */
    readonly type: "node";
    /**
     * The name (also known as "tag name") of this node
     *
     * @type {Identifier}
     */
    name: Identifier;
    /**
     * Tag attached to this value, if any
     *
     * @type {Identifier | null}
     */
    tag: Identifier | null;
    /**
     * Entries of the node
     *
     * @type {Entry[]}
     */
    entries: Entry[];
    /**
     * Children of the node
     *
     * An empty array means the children block is present but empty,
     * if the value is `null` then there is no children block.
     *
     * @type {Document | null}
     */
    children: Document | null;
    /**
     * Leading whitespace
     *
     * @type {string=}
     */
    leading: string | undefined;
    /**
     * Trailing whitespace
     *
     * @type {string=}
     */
    trailing: string | undefined;
    /**
     * Whitespace between the last entry and the children
     *
     * @type {string=}
     */
    beforeChildren: string | undefined;
    /**
     * Create an identical copy of this node
     *
     * @returns {Node}
     */
    clone(): Node;
    /**
     * Return the tag of this node, if any
     *
     * @returns {string | null}
     */
    getTag(): string | null;
    /**
     * Set the tag of this node to the given tag
     *
     * @param {string | null | undefined} tag
     */
    setTag(tag: string | null | undefined): void;
    /**
     * Return the name of this node
     *
     * @returns {string}
     */
    getName(): string;
    /**
     * Set the name of this node to the given name
     *
     * @param {string} name
     */
    setName(name: string): void;
    /**
     * Return whether this node has arguments
     *
     * @returns {boolean}
     */
    hasArguments(): boolean;
    /**
     * Return a snapshot of all arguments of this node
     *
     * Changes to the returned array are not reflected back onto this node
     * itself, and updates to the node won't reflect in the returned array.
     *
     * @returns {Value['value'][]}
     */
    getArguments(): Value['value'][];
    /**
     * Return a snapshot of all arguments of this node
     *
     * Changes to the returned array are not reflected back onto this node
     * itself, and updates to the node won't reflect in the returned array.
     *
     * @returns {Entry[]}
     */
    getArgumentEntries(): Entry[];
    /**
     * Return the value at the given index, if present
     *
     * This index counts towards the arguments only, i.e. if the node has five
     * entries, three of which are arguments then passing `1` returns the second
     * argument, regardless of the whether properties and arguments are
     * interspersed or not.
     *
     * @param {number} index
     * @returns {boolean}
     */
    hasArgument(index: number): boolean;
    /**
     * Return the argument at the given index, if present
     *
     * This index counts towards the arguments only, i.e. if the node has five
     * entries, three of which are arguments then passing `1` returns the second
     * argument, regardless of the whether properties and arguments are
     * interspersed or not.
     *
     * @param {number} index
     * @returns {Value['value'] | undefined}
     */
    getArgument(index: number): Value['value'] | undefined;
    /**
     * Return the argument entry at the given index, if present
     *
     * This index counts towards the arguments only, i.e. if the node has five
     * entries, three of which are arguments then passing `1` returns the second
     * argument, regardless of the whether properties and arguments are
     * interspersed or not.
     *
     * @param {number} index
     * @returns {Entry | undefined}
     */
    getArgumentEntry(index: number): Entry | undefined;
    /**
     * Add the given value as argument to this node
     *
     * The argument is added at the given index, or at the end.
     * This index counts towards the arguments only, i.e. if the node has five
     * entries, three of which are arguments then inserting an argument between
     * the second and third can be achieved by passing `2` regardless of the
     * whether properties and arguments are interspersed or not.
     *
     * @param {Value['value']} value The value to insert as argument
     * @param {string | null} [tag] The tag to attach to the argument, if any
     * @param {number} [index] The index
     */
    addArgument(value: Value['value'], tag?: string | null | undefined, index?: number | undefined): void;
    /**
     * Remove the argument at the given index
     *
     * The index counts towards the arguments only, i.e. if the node has five
     * entries, three of which are arguments then the last argument can be
     * removed by passing `2`, regardless of whether the third argument is also
     * the third entry.
     *
     * @param {number} index
     */
    removeArgument(index: number): void;
    /**
     * Return whether this node has properties
     *
     * @returns {boolean}
     */
    hasProperties(): boolean;
    /**
     * Return a snapshot of all properties of this node
     *
     * Changes to the returned object are not reflected back onto this node
     * itself, and updates to the node won't reflect in the returned object.
     *
     * @returns {Map<string, Value['value']>}
     */
    getProperties(): Map<string, Value['value']>;
    /**
     * Return a snapshot of all properties of this node
     *
     * Changes to the returned array are not reflected back onto this node
     * itself, and updates to the node won't reflect in the returned array.
     *
     * @returns {Entry[]}
     */
    getPropertyEntries(): Entry[];
    /**
     * Return whether this node has the given property
     *
     * @param {string} name
     * @returns {boolean}
     */
    hasProperty(name: string): boolean;
    /**
     * Return the value of the property with the given name, or undefined
     * if it doesn't exist.
     *
     * @param {string} name
     * @returns {Value['value'] | undefined}
     */
    getProperty(name: string): Value['value'] | undefined;
    /**
     * Return the property entry with the given name, or undefined if it doesn't
     * exist.
     *
     * @param {string} name
     * @returns {Entry | undefined}
     */
    getPropertyEntry(name: string): Entry | undefined;
    /**
     * Set the given property on this node
     *
     * This function updates the property entry with the given name,
     * if it exists.
     *
     * @param {string} name
     * @param {Value['value']} value
     * @param {string | null} [tag]
     */
    setProperty(name: string, value: Value['value'], tag?: string | null | undefined): void;
    /**
     * Delete the property with the given name
     *
     * @param {string} name
     */
    deleteProperty(name: string): void;
    /**
     * Return whether this node has child nodes
     *
     * @returns {boolean}
     */
    hasChildren(): boolean;
    /**
     * Add the given node at the end of this node's children
     *
     * @param {Node | Document} node
     */
    appendNode(node: Node | Document): void;
    /**
     * Insert the given node to the node's children before the referenceNode, or at the end if no reference is passed
     *
     * @param {Node | Document} newNode
     * @param {Node | null} referenceNode
     * @throws {Error} If the given referenceNode is not part of this node's children
     */
    insertNodeBefore(newNode: Node | Document, referenceNode: Node | null): void;
    /**
     * Insert the given node to the node's children after the referenceNode, or at the beginning if no reference is passed
     *
     * @param {Node | Document} newNode
     * @param {Node | null} referenceNode
     * @throws {Error} If the given referenceNode is not part of this document
     */
    insertNodeAfter(newNode: Node | Document, referenceNode: Node | null): void;
    /**
     * Remove the given node from this node's children
     *
     * @param {Node} node
     * @throws {Error} if the given node is not in this node's children
     */
    removeNode(node: Node): void;
    /**
     * Replace the old node with the new node in this node's children
     *
     * @param {Node} oldNode
     * @param {Node | Document} newNode
     * @throws {Error} if the oldNode is not in this node's children
     */
    replaceNode(oldNode: Node, newNode: Node | Document): void;
    /**
     * Return all nodes with the given node name
     *
     * Changes to the returned array are not reflected back onto this document
     * itself, and updates to the document won't reflect in the returned array.
     *
     * @param {string} name
     * @returns {Node[]}
     */
    findNodesByName(name: string): Node[];
    /**
     * Return the last node in this node's children with the given name
     *
     * This function returns the last node instead of first to be in line with
     * how properties are defined in the KDL specification where the last
     * property with the given name is used and the rest is shadowed.
     *
     * @param {string} name
     * @returns {Node | undefined}
     */
    findNodeByName(name: string): Node | undefined;
    /**
     * Return the last node in this node's children with the given name, matching the parameter
     *
     * If the parameter is `undefined`, this method looks for a node with any single
     * arguments. If a parameter is passed, this method looks for a node with
     * a single parameter, equal to the given parameter.
     *
     * @param {string} name
     * @param {Value['value']} [parameter]
     * @returns {Node | undefined}
     */
    findParameterizedNode(name: string, parameter?: string | number | boolean | null | undefined): Node | undefined;
    /**
     * Remove all nodes with the given name from this document
     *
     * @param {string} name
     * @returns {void}
     */
    removeNodesByName(name: string): void;
}
import { Identifier } from "./identifier.js";
import { Entry } from "./entry.js";
import { Document } from "./document.js";
import { Value } from "./value.js";
