"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "Discharge", {
    enumerable: true,
    get: function() {
        return Discharge;
    }
});
const _sequelize = require("sequelize");
const _constants = require("@tamanu/constants");
const _errors = require("@tamanu/shared/errors");
const _Model = require("./Model");
const _buildEncounterLinkedSyncFilter = require("../sync/buildEncounterLinkedSyncFilter");
const _buildEncounterLinkedLookupFilter = require("../sync/buildEncounterLinkedLookupFilter");
const _Facility = require("./Facility");
const _logging = require("@tamanu/shared/services/logging");
let Discharge = class Discharge extends _Model.Model {
    static initModel({ primaryKey, ...options }) {
        super.init({
            id: primaryKey,
            note: {
                type: _sequelize.DataTypes.TEXT,
                allowNull: true
            },
            facilityName: {
                type: _sequelize.DataTypes.STRING,
                allowNull: true
            },
            facilityAddress: {
                type: _sequelize.DataTypes.STRING,
                allowNull: true
            },
            facilityTown: {
                type: _sequelize.DataTypes.STRING,
                allowNull: true
            }
        }, {
            ...options,
            syncDirection: _constants.SYNC_DIRECTIONS.BIDIRECTIONAL,
            validate: {
                mustHaveEncounter () {
                    if (!this.deletedAt && !this.encounterId) {
                        throw new _errors.InvalidOperationError('A discharge must have an encounter.');
                    }
                }
            }
        });
    }
    async address() {
        const encounterFacility = await this.sequelize.query(`
        SELECT f.* FROM encounters e
        JOIN locations l on l.id = e.location_id
        JOIN facilities f on f.id = l.facility_id
        WHERE e.id = $encounterId
        `, {
            type: _sequelize.QueryTypes.SELECT,
            model: _Facility.Facility,
            mapToModel: true,
            bind: {
                encounterId: this.encounterId
            }
        }).then((res)=>res?.[0], (err)=>{
            _logging.log.warn('Failed to fetch encounter facility', err);
            return null;
        });
        return {
            name: encounterFacility?.name ?? this.facilityName,
            address: encounterFacility?.streetAddress ?? this.facilityAddress,
            town: encounterFacility?.cityTown ?? this.facilityTown
        };
    }
    static getFullReferenceAssociations() {
        return [
            'discharger',
            'disposition'
        ];
    }
    static initRelations(models) {
        this.belongsTo(models.Encounter, {
            foreignKey: 'encounterId',
            as: 'encounter'
        });
        this.belongsTo(models.User, {
            foreignKey: 'dischargerId',
            as: 'discharger'
        });
        this.belongsTo(models.ReferenceData, {
            foreignKey: 'dispositionId',
            as: 'disposition'
        });
    }
    static buildPatientSyncFilter(patientCount, markedForSyncPatientsTable) {
        if (patientCount === 0) {
            return null;
        }
        return (0, _buildEncounterLinkedSyncFilter.buildEncounterLinkedSyncFilter)([
            this.tableName,
            'encounters'
        ], markedForSyncPatientsTable);
    }
    static async buildSyncLookupQueryDetails() {
        return (0, _buildEncounterLinkedLookupFilter.buildEncounterLinkedLookupFilter)(this);
    }
};

//# sourceMappingURL=Discharge.js.map