"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "adminRoutes", {
    enumerable: true,
    get: function() {
        return adminRoutes;
    }
});
const _express = /*#__PURE__*/ _interop_require_default(require("express"));
const _expressasynchandler = /*#__PURE__*/ _interop_require_default(require("express-async-handler"));
const _middleware = require("@tamanu/shared/permissions/middleware");
const _errors = require("@tamanu/errors");
const _crudHelpers = require("@tamanu/shared/utils/crudHelpers");
const _settings = require("@tamanu/settings");
const _exporter = require("./exporter");
const _importer = require("./importer");
const _asset = require("./asset");
const _fhirJobStats = require("./fhirJobStats");
const _patientMerge = require("./patientMerge");
const _template = require("./template");
const _reportRoutes = require("./reports/reportRoutes");
const _sync = require("./sync");
const _translation = require("./translation");
const _users = require("./users");
function _interop_require_default(obj) {
    return obj && obj.__esModule ? obj : {
        default: obj
    };
}
const adminRoutes = _express.default.Router();
adminRoutes.use(_middleware.ensurePermissionCheck);
adminRoutes.use('/reports', _reportRoutes.reportsRouter);
adminRoutes.use('/translation', _translation.translationRouter);
adminRoutes.post('/mergePatient', _patientMerge.mergePatientHandler);
// A temporary lookup-patient-by-displayId endpoint, just to
// support patient merge because the patient search functionality is only
// available on Facility and there was some time pressure to get it out the door.
// This should be replaced by the full-fledged patient search once some
// more consideration has been put into how that functionality should best
// be shared between the server modules.
adminRoutes.get('/lookup/patient/:displayId', (0, _expressasynchandler.default)(async (req, res)=>{
    req.checkPermission('read', 'Patient');
    const { Patient } = req.store.models;
    const { displayId } = req.params;
    const patient = await Patient.findOne({
        where: {
            displayId
        },
        include: [
            'village'
        ]
    });
    if (!patient) throw new _errors.NotFoundError(`Could not find patient with display ID ${displayId}.`);
    await req.audit.access({
        recordId: patient.id,
        params: req.params,
        model: Patient
    });
    res.send(patient);
}));
adminRoutes.use('/import', _importer.importerRouter);
adminRoutes.use('/export', _exporter.exporterRouter);
adminRoutes.get('/programs', (0, _crudHelpers.simpleGetList)('Program'));
adminRoutes.get('/sync/lastCompleted', _sync.syncLastCompleted);
adminRoutes.get('/fhir/jobStats', _fhirJobStats.fhirJobStats);
adminRoutes.use('/template', _template.templateRoutes);
adminRoutes.use('/asset', _asset.assetRoutes);
adminRoutes.use('/users', _users.usersRouter);
// These settings endpoints are setup for viewing and saving the settings in the JSON editor in the admin panel
adminRoutes.get('/settings', (0, _expressasynchandler.default)(async (req, res)=>{
    req.checkPermission('read', 'Setting');
    const { Setting } = req.store.models;
    const data = await Setting.get('', req.query.facilityId, req.query.scope);
    res.send(data);
}));
adminRoutes.put('/settings', (0, _expressasynchandler.default)(async (req, res)=>{
    req.checkPermission('write', 'Setting');
    const { Setting } = req.store.models;
    await Setting.set('', req.body.settings, req.body.scope, req.body.facilityId);
    res.json({
        code: 200
    });
}));
adminRoutes.delete('/settings/cache', (0, _expressasynchandler.default)(async (req, res)=>{
    req.checkPermission('manage', 'all');
    _settings.settingsCache.reset();
    res.status(204).send();
}));
adminRoutes.get('/facilities', (0, _expressasynchandler.default)(async (req, res)=>{
    req.checkPermission('list', 'Facility');
    const { Facility } = req.store.models;
    const data = await Facility.findAll({
        attributes: [
            'id',
            'name'
        ]
    });
    res.send(data);
}));

//# sourceMappingURL=adminRoutes.js.map