"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
function _export(target, all) {
    for(var name in all)Object.defineProperty(target, name, {
        enumerable: true,
        get: all[name]
    });
}
_export(exports, {
    userInfo: function() {
        return userInfo;
    },
    userMiddleware: function() {
        return userMiddleware;
    }
});
const _api = require("@opentelemetry/api");
const _expressasynchandler = /*#__PURE__*/ _interop_require_default(require("express-async-handler"));
const _config = /*#__PURE__*/ _interop_require_default(require("config"));
const _auth = require("@tamanu/constants/auth");
const _errors = require("@tamanu/errors");
const _utils = require("./utils");
const _createSessionIdentifier = require("@tamanu/shared/audit/createSessionIdentifier");
const _packagejson = require("../../package.json");
function _interop_require_default(obj) {
    return obj && obj.__esModule ? obj : {
        default: obj
    };
}
const userMiddleware = ({ secret })=>(0, _expressasynchandler.default)(async (req, res, next)=>{
        const { store, headers, settings } = req;
        const { canonicalHostName } = _config.default;
        // get token
        const { authorization } = headers;
        if (!authorization) {
            throw new _errors.MissingCredentialError('Missing authorization header');
        }
        // verify token
        const [bearer, token] = authorization.split(/\s/);
        const sessionId = (0, _createSessionIdentifier.createSessionIdentifier)(token);
        if (bearer.toLowerCase() !== 'bearer') {
            throw new _errors.InvalidCredentialError('Only Bearer token is supported');
        }
        let contents = null;
        try {
            contents = await (0, _utils.verifyToken)(token, secret, {
                issuer: canonicalHostName,
                audience: _auth.JWT_TOKEN_TYPES.ACCESS
            });
        } catch (e) {
            throw new _errors.InvalidTokenError();
        }
        const { userId, deviceId } = contents;
        const user = await (0, _utils.findUserById)(store.models, userId);
        if (!user) {
            throw new _errors.InvalidTokenError('User specified in token does not exist').withExtraData({
                userId
            });
        }
        const device = deviceId && await store.models.Device.findByPk(deviceId);
        /* eslint-disable require-atomic-updates */ // in this case we don't care if we're overwriting the user/deviceId
        // and express also guarantees execution order for middlewares
        req.user = user;
        req.deviceId = deviceId;
        req.device = device;
        req.sessionId = sessionId;
        /* eslint-enable require-atomic-updates */ const auditSettings = await settings?.[req.facilityId]?.get('audit');
        // Auditing middleware
        // eslint-disable-next-line require-atomic-updates
        req.audit = {
            access: async ({ recordId, params, model })=>{
                if (!auditSettings?.accesses.enabled) return;
                return req.models.AccessLog.create({
                    userId,
                    recordId,
                    recordType: model.name,
                    sessionId,
                    isMobile: false,
                    frontEndContext: params,
                    backEndContext: {
                        endpoint: req.originalUrl
                    },
                    loggedAt: new Date(),
                    facilityId: null,
                    deviceId: req.deviceId || 'unknown-device',
                    version: _packagejson.version
                });
            }
        };
        const spanAttributes = user ? {
            'app.user.id': user.id,
            'app.user.role': user.role
        } : {};
        // eslint-disable-next-line no-unused-expressions
        _api.trace.getActiveSpan()?.setAttributes(spanAttributes);
        _api.context.with(_api.propagation.setBaggage(_api.context.active(), _api.propagation.createBaggage(spanAttributes)), ()=>next());
    });
const userInfo = (0, _expressasynchandler.default)(async (req, res)=>{
    if (!req.user) {
        throw new _errors.ForbiddenError();
    }
    res.send((0, _utils.stripUser)(req.user));
});

//# sourceMappingURL=userMiddleware.js.map