import { DataTypes, Op, Sequelize } from 'sequelize';
import { DEVICE_SCOPES, DEVICE_SCOPES_SUBJECT_TO_QUOTA, SYNC_DIRECTIONS } from '@tamanu/constants';
import { InvalidOperationError } from '@tamanu/errors';
import { Model } from './Model';
export class Device extends Model {
    // eslint-disable-next-line @typescript-eslint/no-unused-vars
    static initModel({ primaryKey, ...options }) {
        super.init({
            id: {
                type: DataTypes.TEXT,
                allowNull: false,
                primaryKey: true,
                defaultValue: Sequelize.fn('gen_random_uuid')
            },
            lastSeenAt: {
                type: DataTypes.DATE,
                allowNull: false,
                defaultValue: Sequelize.fn('now')
            },
            scopes: {
                type: DataTypes.JSONB,
                allowNull: false,
                defaultValue: []
            },
            name: {
                type: DataTypes.TEXT
            }
        }, {
            ...options,
            syncDirection: SYNC_DIRECTIONS.DO_NOT_SYNC,
            validate: {
                mustHaveKnownScopes () {
                    if (// `as` necessary because of incomplete typings in sequelize
                    this.scopes.some((scope)=>!Object.values(DEVICE_SCOPES).includes(scope))) {
                        throw new InvalidOperationError('Device must only use known scopes.');
                    }
                }
            }
        });
    }
    static initRelations(models) {
        this.belongsTo(models.User, {
            foreignKey: 'registeredById',
            as: 'registeredBy'
        });
    }
    static async getQuotaByUserId(userId) {
        return this.count({
            where: {
                registeredById: userId,
                [Op.or]: DEVICE_SCOPES_SUBJECT_TO_QUOTA.map((scope)=>// the jsonb operator `?`: "Does the string exist as a top-level key within the JSON value?"
                    Sequelize.literal(`scopes ? '${scope}'`))
            }
        });
    }
    async markSeen() {
        await this.update({
            lastSeenAt: Sequelize.fn('now')
        });
    }
    requiresQuota() {
        return this.scopes.some((scope)=>DEVICE_SCOPES_SUBJECT_TO_QUOTA.includes(scope));
    }
    ensureHasScope(scope) {
        if (!this.scopes.includes(scope)) {
            throw new Error('Device must have the required scope.');
        }
    }
}

//# sourceMappingURL=Device.js.map