"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
function _export(target, all) {
    for(var name in all)Object.defineProperty(target, name, {
        enumerable: true,
        get: all[name]
    });
}
_export(exports, {
    ensureRequiredQuestionsPresent: function() {
        return ensureRequiredQuestionsPresent;
    },
    validateChartingSurvey: function() {
        return validateChartingSurvey;
    },
    validateVitalsSurvey: function() {
        return validateVitalsSurvey;
    }
});
const _sequelize = require("sequelize");
const _constants = require("@tamanu/constants");
const _errors = require("../errors");
const REQUIRED_QUESTION_IDS = {
    [_constants.SURVEY_TYPES.VITALS]: Object.values(_constants.VITALS_DATA_ELEMENT_IDS)
};
function ensureRequiredQuestionsPresent(surveyInfo, questionRecords) {
    const { surveyType, sheetName } = surveyInfo;
    const requiredQuestions = REQUIRED_QUESTION_IDS[surveyType];
    if (!requiredQuestions) {
        return;
    }
    // check that each mandatory question ID has a corresponding row in the import sheet
    const hasQuestion = (id)=>questionRecords.some((q)=>q.values.id === id);
    const missingQuestions = requiredQuestions.filter((rf)=>!hasQuestion(rf));
    if (missingQuestions.length > 0) {
        throw new _errors.ValidationError(sheetName, -2, `Survey missing required questions: ${missingQuestions.join(', ')}`);
    }
}
async function ensureOnlyOneVitalsSurveyExists({ models }, surveyInfo) {
    const vitalsCount = await models.Survey.count({
        where: {
            id: {
                [_sequelize.Op.not]: surveyInfo.id
            },
            surveyType: _constants.SURVEY_TYPES.VITALS
        }
    });
    if (vitalsCount > 0) {
        throw new _errors.ImporterMetadataError('Only one vitals survey may exist at a time');
    }
}
function ensureSurveyNonSensitive(surveyInfo, errorMessage) {
    if (surveyInfo.isSensitive) {
        throw new _errors.ImporterMetadataError(errorMessage);
    }
}
async function validateVitalsSurvey(context, surveyInfo) {
    await ensureOnlyOneVitalsSurveyExists(context, surveyInfo);
    ensureSurveyNonSensitive(surveyInfo, 'Vitals survey can not be sensitive');
}
async function ensureOnlyOneComplexSurveySetPerProgram({ models }, surveyInfo) {
    const surveyCount = await models.Survey.count({
        where: {
            id: {
                [_sequelize.Op.not]: surveyInfo.id
            },
            surveyType: surveyInfo.surveyType,
            programId: surveyInfo.programId
        }
    });
    if (surveyCount > 0) {
        throw new _errors.ImporterMetadataError('Complex chart set already exists for this program');
    }
}
async function validateChartingSurvey(context, surveyInfo) {
    if (surveyInfo.surveyType !== _constants.SURVEY_TYPES.SIMPLE_CHART) {
        await ensureOnlyOneComplexSurveySetPerProgram(context, surveyInfo);
    }
    ensureSurveyNonSensitive(surveyInfo, 'Charting survey can not be sensitive');
}

//# sourceMappingURL=validation.js.map