"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "survey", {
    enumerable: true,
    get: function() {
        return survey;
    }
});
const _express = /*#__PURE__*/ _interop_require_default(require("express"));
const _expressasynchandler = /*#__PURE__*/ _interop_require_default(require("express-async-handler"));
const _sequelize = require("sequelize");
const _constants = require("@tamanu/constants");
const _getFilteredListByPermission = require("@tamanu/shared/utils/getFilteredListByPermission");
const _errors = require("@tamanu/errors");
const _crudHelpers = require("@tamanu/shared/utils/crudHelpers");
const _ability = require("@casl/ability");
function _interop_require_default(obj) {
    return obj && obj.__esModule ? obj : {
        default: obj
    };
}
const survey = _express.default.Router();
// There should only be one survey with surveyType vitals, fetch it
// Needs to be added before the /:id endpoint so that endpoint doesn't catch it instead
survey.get('/vitals', (0, _expressasynchandler.default)(async (req, res)=>{
    const { models } = req;
    req.checkPermission('read', 'Vitals');
    const surveyRecord = await models.Survey.findOne({
        where: {
            surveyType: 'vitals'
        }
    });
    if (!surveyRecord) throw new _errors.NotFoundError();
    const components = await models.SurveyScreenComponent.getComponentsForSurvey(surveyRecord.id, {
        includeAllVitals: true
    });
    res.send({
        ...surveyRecord.forResponse(),
        components
    });
}));
survey.get('/charts', (0, _expressasynchandler.default)(async (req, res)=>{
    req.flagPermissionChecked();
    const { models: { Survey } } = req;
    const chartSurveys = await Survey.findAll({
        where: {
            [_sequelize.Op.or]: [
                // Get all current simple and complex charts
                {
                    surveyType: {
                        [_sequelize.Op.in]: [
                            _constants.SURVEY_TYPES.SIMPLE_CHART,
                            _constants.SURVEY_TYPES.COMPLEX_CHART
                        ]
                    },
                    visibilityStatus: _constants.VISIBILITY_STATUSES.CURRENT
                },
                // Get all historical simple and complex charts with answers
                {
                    [_sequelize.Op.and]: [
                        {
                            surveyType: {
                                [_sequelize.Op.in]: [
                                    _constants.SURVEY_TYPES.SIMPLE_CHART,
                                    _constants.SURVEY_TYPES.COMPLEX_CHART
                                ]
                            },
                            visibilityStatus: _constants.VISIBILITY_STATUSES.HISTORICAL
                        },
                        (0, _sequelize.literal)(`EXISTS (SELECT 1 FROM survey_responses WHERE survey_responses.survey_id = "Survey".id AND survey_responses.deleted_at IS NULL)`)
                    ]
                },
                // Get all complex core charts regardless of visibility status
                {
                    surveyType: _constants.SURVEY_TYPES.COMPLEX_CHART_CORE
                }
            ]
        },
        order: [
            [
                'name',
                'ASC'
            ]
        ]
    });
    const permittedChartSurveys = chartSurveys.filter((survey)=>req.ability.can('list', (0, _ability.subject)('Charting', {
            id: survey.id
        })));
    res.send(permittedChartSurveys);
}));
survey.get('/procedureType/:procedureTypeId', (0, _expressasynchandler.default)(async (req, res)=>{
    const { models, ability, params } = req;
    const { procedureTypeId } = params;
    const { ProcedureTypeSurvey } = models;
    req.checkPermission('list', 'Survey');
    const procedureTypeSurveys = await ProcedureTypeSurvey.findAll({
        where: {
            procedureTypeId: procedureTypeId
        },
        include: [
            {
                model: models.Survey,
                as: 'survey',
                where: {
                    visibilityStatus: {
                        [_sequelize.Op.ne]: _constants.VISIBILITY_STATUSES.HISTORICAL
                    }
                }
            }
        ],
        order: [
            [
                'survey',
                'name',
                'ASC'
            ]
        ]
    });
    // Extract the surveys from the join results
    const surveys = procedureTypeSurveys.map((pts)=>pts.survey);
    const filteredSurveys = (0, _getFilteredListByPermission.getFilteredListByPermission)(ability, surveys, 'submit');
    res.send(filteredSurveys);
}));
survey.get('/:id', (0, _expressasynchandler.default)(async (req, res)=>{
    const { models, params } = req;
    const surveyRecord = await (0, _crudHelpers.findRouteObject)(req, 'Survey');
    const components = await models.SurveyScreenComponent.getComponentsForSurvey(params.id, {
        includeAllVitals: true
    });
    res.send({
        ...surveyRecord.forResponse(),
        components
    });
}));
survey.get('/$', (0, _expressasynchandler.default)(async (req, res)=>{
    const { models, ability } = req;
    req.checkPermission('list', 'Survey');
    const surveys = await models.Survey.findAll({
        where: {
            surveyType: req.query.type,
            visibilityStatus: {
                [_sequelize.Op.ne]: _constants.VISIBILITY_STATUSES.HISTORICAL
            }
        },
        order: [
            [
                'name',
                'ASC'
            ]
        ]
    });
    const filteredSurveys = (0, _getFilteredListByPermission.getFilteredListByPermission)(ability, surveys, 'submit');
    res.send({
        surveys: filteredSurveys
    });
}));
const surveyRelations = (0, _crudHelpers.permissionCheckingRouter)('list', 'SurveyResponse');
surveyRelations.get('/:id/surveyResponses', (0, _expressasynchandler.default)(async (req, res)=>{
    const { id: surveyId } = req.params;
    const survey = await req.models.Survey.findByPk(surveyId);
    req.checkPermission('read', survey);
    const response = await (0, _crudHelpers.getResourceList)(req, 'SurveyResponse', 'surveyId');
    res.send(response);
}));
survey.use(surveyRelations);

//# sourceMappingURL=survey.js.map