"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "ReportRequestProcessor", {
    enumerable: true,
    get: function() {
        return ReportRequestProcessor;
    }
});
const _config = /*#__PURE__*/ _interop_require_default(require("config"));
const _sequelize = /*#__PURE__*/ _interop_require_default(require("sequelize"));
const _child_process = require("child_process");
const _constants = require("@tamanu/constants");
const _reports = require("@tamanu/shared/reports");
const _tasks = require("@tamanu/shared/tasks");
const _logging = require("@tamanu/shared/services/logging");
const _ReportRunner = require("../report/ReportRunner");
const _localisation = require("../localisation");
function _define_property(obj, key, value) {
    if (key in obj) {
        Object.defineProperty(obj, key, {
            value: value,
            enumerable: true,
            configurable: true,
            writable: true
        });
    } else {
        obj[key] = value;
    }
    return obj;
}
function _interop_require_default(obj) {
    return obj && obj.__esModule ? obj : {
        default: obj
    };
}
let ReportRequestProcessor = class ReportRequestProcessor extends _tasks.ScheduledTask {
    getName() {
        return 'ReportRequestProcessor';
    }
    registerExitListeners() {
        const killChildProcesses = (event)=>{
            this.childProcesses.forEach((childProcess)=>{
                if (!childProcess?.kill || childProcess.killed) return;
                childProcess.kill(childProcess.pid, event);
                _logging.log.info('Cleaned up child process that was not killed by the parent process');
            });
        };
        process.on([
            'uncaughtException',
            'SIGINT',
            'SIGTERM'
        ], killChildProcesses);
    }
    async countQueue() {
        return this.context.store.models.ReportRequest.count({
            where: {
                status: _constants.REPORT_REQUEST_STATUSES.RECEIVED
            }
        });
    }
    async runReports() {
        const localisation = await (0, _localisation.getLocalisation)();
        const requests = await this.context.store.models.ReportRequest.findAll({
            where: {
                status: _constants.REPORT_REQUEST_STATUSES.RECEIVED
            },
            order: [
                [
                    'createdAt',
                    'ASC'
                ]
            ],
            limit: this.config.limit
        });
        for (const request of requests){
            const reportId = request.getReportId();
            if (!_config.default.mailgun.from) {
                _logging.log.error(`ReportRequestProcessorError - Email config missing`);
                await request.update({
                    status: _constants.REPORT_REQUEST_STATUSES.ERROR,
                    error: 'Email config missing'
                });
                return;
            }
            const { disabledReports } = localisation;
            if (disabledReports.includes(reportId)) {
                _logging.log.error(`Report "${reportId}" is disabled`);
                await request.update({
                    status: _constants.REPORT_REQUEST_STATUSES.ERROR,
                    error: `Report "${reportId}" is disabled`
                });
                return;
            }
            const reportModule = await (0, _reports.getReportModule)(reportId, this.context.store.models);
            const reportDataGenerator = reportModule?.dataGenerator;
            if (!reportModule || !reportDataGenerator) {
                _logging.log.error(`ReportRequestProcessorError - Unable to find report generator for report ${request.id} of type ${reportId}`);
                await request.update({
                    status: _constants.REPORT_REQUEST_STATUSES.ERROR,
                    error: `Unable to find report generator for report ${request.id} of type ${reportId}`
                });
                return;
            }
            try {
                await request.update({
                    status: _constants.REPORT_REQUEST_STATUSES.PROCESSING,
                    processStartedTime: new Date()
                });
                const shouldRunInChildProcess = await this.context.settings.get('reportProcess.runInChildProcess');
                const runReport = shouldRunInChildProcess ? this.spawnReportProcess : this.runReportInTheSameProcess;
                await runReport(request);
                await request.update({
                    status: _constants.REPORT_REQUEST_STATUSES.PROCESSED
                });
            } catch (e) {
                _logging.log.error(`${e.stack}\nReportRequestProcessorError - Failed to generate report`);
                await request.update({
                    status: _constants.REPORT_REQUEST_STATUSES.ERROR,
                    error: e.stack
                });
            }
        }
    }
    async validateTimeoutReports() {
        try {
            const timeOutDurationSeconds = await this.context.settings.get('reportProcess.timeOutDurationSeconds');
            const requests = await this.context.store.models.ReportRequest.findAll({
                where: _sequelize.default.literal(`status = '${_constants.REPORT_REQUEST_STATUSES.PROCESSING}' AND
          EXTRACT(EPOCH FROM CURRENT_TIMESTAMP - process_started_time) > ${timeOutDurationSeconds}`),
                order: [
                    [
                        'createdAt',
                        'ASC'
                    ]
                ],
                limit: 10
            });
            for (const request of requests){
                _logging.log.info(`ReportRequestProcessorError - Marking report request "${request.id}" as timed out`);
                await request.update({
                    status: _constants.REPORT_REQUEST_STATUSES.ERROR,
                    error: 'Report timed out'
                });
            }
        } catch (error) {
            _logging.log.error('ReportRequestProcessorError - Error checking processing reports', error);
        }
    }
    async run() {
        await this.validateTimeoutReports();
        await this.runReports();
    }
    constructor(context){
        // run at 30 seconds interval, process 10 report requests each time
        const conf = _config.default.schedules.reportRequestProcessor;
        const { schedule, jitterTime, enabled } = conf;
        super(schedule, _logging.log, jitterTime, enabled), _define_property(this, "spawnReportProcess", async (request)=>{
            const [node, scriptPath] = process.argv;
            const { childProcessEnv, processOptions, timeOutDurationSeconds } = await this.context.settings.get('reportProcess');
            const parameters = processOptions || process.execArgv;
            _logging.log.info(`Spawning child process for report request "${request.id}" for report "${request.getReportId()}" with command [${node}, ${parameters.toString()}, ${scriptPath}].`);
            // For some reasons, when running a child process under pm2, pm2_env was not set and caused a problem.
            // So this is a work around
            const childEnv = childProcessEnv || {
                ...process.env,
                pm2_env: JSON.stringify(process.env)
            };
            const sleepAfterReport = await this.context.settings.get('reportProcess.sleepAfterReport');
            const childProcess = (0, _child_process.spawn)(node, [
                ...parameters,
                scriptPath,
                'report',
                '--reportId',
                request.getReportId(),
                '--parameters',
                request.parameters,
                '--recipients',
                request.recipients,
                '--userId',
                request.requestedByUserId,
                '--format',
                request.exportFormat,
                '--sleepAfterReport',
                JSON.stringify(sleepAfterReport)
            ], {
                // Time out and kill the report process if it takes too long to run (default 2 hours)
                timeout: timeOutDurationSeconds * 1000,
                env: childEnv
            });
            let errorMessage = '';
            const captureErrorOutput = (message)=>{
                if (message.startsWith('Report failed:')) {
                    errorMessage = message;
                }
            };
            return new Promise((resolve, reject)=>{
                childProcess.on('exit', (code)=>{
                    this.childProcesses.delete(childProcess.pid);
                    if (code === 0) {
                        _logging.log.info(`Child process running report request "${request.id}" for report "${request.getReportId()}" has finished.`);
                        resolve();
                        return;
                    }
                    reject(new Error(errorMessage || `Failed to generate report for report request "${request.id}" for report "${request.getReportId()}"`));
                });
                childProcess.on('error', ()=>{
                    reject(new Error(`Child process failed to start, using commands [${node}, ${scriptPath}].`));
                });
                // Catch error from child process
                childProcess.stdout.on('data', (data)=>{
                    captureErrorOutput(data.toString());
                    process.stdout.write(data);
                });
                childProcess.stderr.on('data', (data)=>{
                    captureErrorOutput(data.toString());
                    process.stderr.write(data);
                });
                this.childProcesses.set(childProcess.pid, childProcess);
            });
        }), _define_property(this, "runReportInTheSameProcess", async (request)=>{
            _logging.log.info(`Running report request "${request.id}" for report "${request.getReportId()}" in main process.`);
            const sleepAfterReport = await this.context.settings.get('reportProcess.sleepAfterReport');
            const reportRunner = new _ReportRunner.ReportRunner(request.getReportId(), request.getParameters(), request.getRecipients(), this.context.store, this.context.reportSchemaStores, this.context.emailService, request.requestedByUserId, request.exportFormat, sleepAfterReport);
            await reportRunner.run();
        });
        this.config = conf;
        this.context = context;
        this.childProcesses = new Map();
        this.registerExitListeners();
    }
};

//# sourceMappingURL=ReportRequestProcessor.js.map