"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "Invoice", {
    enumerable: true,
    get: function() {
        return Invoice;
    }
});
const _sequelize = require("sequelize");
const _constants = require("@tamanu/constants");
const _Model = require("./Model");
const _buildEncounterLinkedSyncFilter = require("../sync/buildEncounterLinkedSyncFilter");
const _buildEncounterLinkedLookupFilter = require("../sync/buildEncounterLinkedLookupFilter");
const _model = require("../types/model");
let Invoice = class Invoice extends _Model.Model {
    static initModel({ primaryKey, ...options }) {
        super.init({
            id: primaryKey,
            displayId: {
                type: _sequelize.DataTypes.STRING,
                allowNull: false
            },
            date: (0, _model.dateTimeType)('date', {
                allowNull: false
            }),
            status: {
                type: _sequelize.DataTypes.STRING,
                allowNull: false
            },
            patientPaymentStatus: {
                type: _sequelize.DataTypes.STRING,
                allowNull: false,
                defaultValue: _constants.INVOICE_PATIENT_PAYMENT_STATUSES.UNPAID
            },
            insurerPaymentStatus: {
                type: _sequelize.DataTypes.STRING,
                allowNull: false,
                defaultValue: _constants.INVOICE_INSURER_PAYMENT_STATUSES.UNPAID
            }
        }, {
            ...options,
            syncDirection: _constants.SYNC_DIRECTIONS.BIDIRECTIONAL
        });
    }
    static initRelations(models) {
        this.belongsTo(models.Encounter, {
            foreignKey: 'encounterId',
            as: 'encounter'
        });
        this.hasOne(models.InvoiceDiscount, {
            foreignKey: 'invoiceId',
            as: 'discount'
        });
        this.hasMany(models.InvoiceInsurer, {
            foreignKey: 'invoiceId',
            as: 'insurers'
        });
        this.hasMany(models.InvoiceItem, {
            foreignKey: 'invoiceId',
            as: 'items'
        });
        this.hasMany(models.InvoicePayment, {
            foreignKey: 'invoiceId',
            as: 'payments'
        });
    }
    static buildPatientSyncFilter(patientCount, markedForSyncPatientsTable) {
        if (patientCount === 0) {
            return null;
        }
        return (0, _buildEncounterLinkedSyncFilter.buildEncounterLinkedSyncFilter)([
            this.tableName,
            'encounters'
        ], markedForSyncPatientsTable);
    }
    static async buildSyncLookupQueryDetails() {
        return (0, _buildEncounterLinkedLookupFilter.buildEncounterLinkedLookupFilter)(this);
    }
    static getFullReferenceAssociations() {
        const { models } = this.sequelize;
        return [
            'encounter',
            {
                model: models.InvoiceDiscount,
                as: 'discount',
                include: [
                    {
                        model: models.User,
                        as: 'appliedByUser',
                        attributes: [
                            'displayName'
                        ]
                    }
                ]
            },
            {
                model: models.InvoiceInsurer,
                as: 'insurers',
                include: [
                    {
                        model: models.ReferenceData,
                        as: 'insurer'
                    }
                ]
            },
            {
                model: models.InvoiceItem,
                as: 'items',
                include: models.InvoiceItem.getListReferenceAssociations(models)
            },
            {
                model: models.InvoicePayment,
                as: 'payments',
                include: models.InvoicePayment.getListReferenceAssociations(models)
            }
        ];
    }
};

//# sourceMappingURL=Invoice.js.map