"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "getValues", {
    enumerable: true,
    get: function() {
        return getValues;
    }
});
const _constants = require("@tamanu/constants");
const _fhirTypes = require("@tamanu/shared/services/fhirTypes");
const _fhir = require("@tamanu/shared/utils/fhir");
async function getValues(upstream, models) {
    const { AdministeredVaccine } = models;
    if (upstream instanceof AdministeredVaccine) return getValuesFromAdministeredVaccine(upstream);
    throw new Error(`Invalid upstream type for immunization ${upstream.constructor.name}`);
}
async function getValuesFromAdministeredVaccine(administeredVaccine) {
    const { encounter, scheduledVaccine, recorder } = administeredVaccine;
    const { patient } = encounter || {};
    return {
        lastUpdated: new Date(),
        status: status(administeredVaccine.status),
        vaccineCode: vaccineCode(scheduledVaccine),
        patient: patientReference(patient),
        encounter: encounterReference(encounter),
        occurrenceDateTime: (0, _fhir.formatFhirDate)(administeredVaccine.date),
        lotNumber: administeredVaccine.batch,
        site: site(administeredVaccine.injectionSite),
        performer: performer(recorder),
        protocolApplied: protocolApplied(scheduledVaccine?.doseLabel),
        resolved: true
    };
}
function status(administeredVaccineStatus) {
    switch(administeredVaccineStatus){
        case _constants.VACCINE_STATUS.GIVEN:
            return 'completed';
        case _constants.VACCINE_STATUS.RECORDED_IN_ERROR:
            return 'entered-in-error';
        case _constants.VACCINE_STATUS.NOT_GIVEN:
        case _constants.VACCINE_STATUS.SCHEDULED:
        case _constants.VACCINE_STATUS.MISSED:
        case _constants.VACCINE_STATUS.DUE:
        case _constants.VACCINE_STATUS.UPCOMING:
        case _constants.VACCINE_STATUS.OVERDUE:
        case _constants.VACCINE_STATUS.UNKNOWN:
        default:
            return 'not-done';
    }
}
function vaccineCode(scheduledVaccine) {
    function vaccineIdToAIRVCode(vaccineId) {
        switch(vaccineId){
            case KNOWN_VACCINE_IDS.PFIZER:
                return KNOWN_AIRV_CODES.COMIRN;
            case KNOWN_VACCINE_IDS.ASTRAZENECA:
                return KNOWN_AIRV_CODES.COVAST;
            default:
                return null;
        }
    }
    // AIRV: Australian Immunisation Register Vaccine
    const AIRV_TERMINOLOGY_URL = 'https://www.healthterminologies.gov.au/integration/R4/fhir/ValueSet/australian-immunisation-register-vaccine-1';
    // All known vaccines are reference data IDs (type 'drug')
    const KNOWN_VACCINE_IDS = {
        PFIZER: 'drug-COVID-19-Pfizer',
        ASTRAZENECA: 'drug-COVAX'
    };
    // All currently supported AIRV vaccine codes
    const KNOWN_AIRV_CODES = {
        COMIRN: 'COMIRN',
        COVAST: 'COVAST'
    };
    const code = vaccineIdToAIRVCode(scheduledVaccine?.vaccine?.id);
    // Only include a coding if we support the code, otherwise just use text
    return new _fhirTypes.FhirCodeableConcept({
        ...code && {
            coding: [
                new _fhirTypes.FhirCoding({
                    system: AIRV_TERMINOLOGY_URL,
                    code
                })
            ]
        },
        text: scheduledVaccine?.vaccine?.name
    });
}
function patientReference(patient) {
    return new _fhirTypes.FhirReference({
        reference: `Patient/${patient?.id}`,
        display: [
            patient?.firstName,
            patient?.lastName
        ].filter((x)=>x).join(' ')
    });
}
function encounterReference(encounter) {
    return new _fhirTypes.FhirReference({
        reference: `Encounter/${encounter?.id}`
    });
}
function site(injectionSite) {
    const HL7_INJECTION_SITE_URL = 'http://terminology.hl7.org/CodeSystem/v3-ActSite';
    // Dictionary that maps Tamanu injection site to HL7 code
    const INJECTION_SITE_TO_HL7_CODE = {
        [_constants.INJECTION_SITE_VALUES.RIGHT_ARM]: 'RA',
        [_constants.INJECTION_SITE_VALUES.LEFT_ARM]: 'LA',
        [_constants.INJECTION_SITE_VALUES.RIGHT_THIGH]: 'RT',
        [_constants.INJECTION_SITE_VALUES.LEFT_THIGH]: 'LT'
    };
    return [
        new _fhirTypes.FhirCodeableConcept({
            coding: [
                new _fhirTypes.FhirCoding({
                    system: HL7_INJECTION_SITE_URL,
                    code: INJECTION_SITE_TO_HL7_CODE[injectionSite] || null,
                    display: injectionSite
                })
            ]
        })
    ];
}
function performer(recorder) {
    if (!recorder) return [];
    return [
        new _fhirTypes.FhirImmunizationPerformer({
            actor: new _fhirTypes.FhirReference({
                reference: `Practitioner/${recorder.id}`
            })
        })
    ];
}
function protocolApplied(schedule) {
    return [
        new _fhirTypes.FhirImmunizationProtocolApplied({
            doseNumberString: schedule
        })
    ];
}

//# sourceMappingURL=getValues.js.map