/* eslint-disable no-param-reassign */ "use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "dataGenerator", {
    enumerable: true,
    get: function() {
        return dataGenerator;
    }
});
const _sequelize = require("sequelize");
const _datefns = require("date-fns");
const _dateTime = require("@tamanu/utils/dateTime");
const _utilities = require("../utilities");
const DATE_FORMAT = 'yyyy/MM/dd';
const reportColumnTemplate = [
    {
        title: 'Patient Name',
        accessor: (data)=>data.patientName
    },
    {
        title: 'UID',
        accessor: (data)=>data.uid
    },
    {
        title: 'DOB',
        accessor: (data)=>(0, _dateTime.format)(data.dob, DATE_FORMAT)
    },
    {
        title: 'Sex',
        accessor: (data)=>data.sex
    },
    {
        title: 'Village',
        accessor: (data)=>data.village
    },
    {
        title: 'First dose given',
        accessor: (data)=>data.dose1
    },
    {
        title: 'First dose date',
        accessor: (data)=>(0, _dateTime.format)(data.dose1Date, DATE_FORMAT)
    },
    {
        title: 'Second dose given',
        accessor: (data)=>data.dose2
    },
    {
        title: 'Second dose date',
        accessor: (data)=>(0, _dateTime.format)(data.dose2Date, DATE_FORMAT)
    },
    {
        title: 'Vaccine Name',
        accessor: (data)=>data.vaccineLabel
    }
];
function parametersToSqlWhere(parameters) {
    parameters.fromDate = (0, _dateTime.toDateTimeString)((0, _datefns.startOfDay)(parameters.fromDate ? (0, _datefns.parseISO)(parameters.fromDate) : (0, _datefns.subDays)(new Date(), 30)));
    if (parameters.toDate) {
        parameters.toDate = (0, _dateTime.toDateTimeString)((0, _datefns.endOfDay)((0, _datefns.parseISO)(parameters.toDate)));
    }
    const whereClause = Object.entries(parameters).filter(([, val])=>val).reduce((where, [key, value])=>{
        switch(key){
            case 'village':
                where['$encounter->patient.village_id$'] = value;
                break;
            case 'fromDate':
                if (!where.date) {
                    where.date = {};
                }
                where.date[_sequelize.Op.gte] = value;
                break;
            case 'toDate':
                if (!where.date) {
                    where.date = {};
                }
                where.date[_sequelize.Op.lte] = value;
                break;
            default:
                break;
        }
        return where;
    }, {
        '$scheduledVaccine.label$': {
            [_sequelize.Op.in]: [
                'COVID-19 AZ',
                'COVID-19 Pfizer'
            ]
        }
    });
    return whereClause;
}
async function queryCovidVaccineListData(models, parameters) {
    const result = await models.AdministeredVaccine.findAll({
        include: [
            {
                model: models.Encounter,
                as: 'encounter',
                include: [
                    {
                        model: models.Patient,
                        as: 'patient',
                        include: [
                            {
                                model: models.ReferenceData,
                                as: 'village'
                            }
                        ]
                    }
                ]
            },
            {
                model: models.ScheduledVaccine,
                as: 'scheduledVaccine'
            }
        ],
        where: parametersToSqlWhere(parameters)
    });
    const administeredVaccines = result.map((r)=>r.get({
            plain: true
        }));
    const patients = administeredVaccines.reduce((acc, vaccine)=>{
        if (!vaccine.encounter?.patientId) {
            return acc;
        }
        const { encounter: { patientId, patient: { displayId, firstName, lastName, dateOfBirth, village, sex } }, date, scheduledVaccine: { doseLabel, label } } = vaccine;
        if (!acc[patientId]) {
            acc[patientId] = {
                patientName: `${firstName} ${lastName}`,
                uid: displayId,
                dob: (0, _datefns.parseISO)(dateOfBirth).toLocaleDateString(),
                village: village?.name,
                dose1: 'No',
                dose2: 'No',
                vaccineLabel: label,
                sex
            };
        }
        if (doseLabel === 'Dose 1') {
            acc[patientId].dose1 = 'Yes';
            acc[patientId].dose1Date = (0, _datefns.parseISO)(date).toLocaleDateString();
        }
        if (doseLabel === 'Dose 2') {
            acc[patientId].dose2 = 'Yes';
            acc[patientId].dose2Date = (0, _datefns.parseISO)(date).toLocaleDateString();
        }
        return acc;
    }, {});
    return Object.values(patients);
}
async function dataGenerator({ models }, parameters) {
    const queryResults = await queryCovidVaccineListData(models, parameters);
    return (0, _utilities.generateReportFromQueryData)(queryResults, reportColumnTemplate);
}

//# sourceMappingURL=covid-vaccine-list.js.map