"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "Device", {
    enumerable: true,
    get: function() {
        return Device;
    }
});
const _sequelize = require("sequelize");
const _constants = require("@tamanu/constants");
const _errors = require("@tamanu/errors");
const _Model = require("./Model");
let Device = class Device extends _Model.Model {
    // eslint-disable-next-line @typescript-eslint/no-unused-vars
    static initModel({ primaryKey, ...options }) {
        super.init({
            id: {
                type: _sequelize.DataTypes.TEXT,
                allowNull: false,
                primaryKey: true,
                defaultValue: _sequelize.Sequelize.fn('gen_random_uuid')
            },
            lastSeenAt: {
                type: _sequelize.DataTypes.DATE,
                allowNull: false,
                defaultValue: _sequelize.Sequelize.fn('now')
            },
            scopes: {
                type: _sequelize.DataTypes.JSONB,
                allowNull: false,
                defaultValue: []
            },
            name: {
                type: _sequelize.DataTypes.TEXT
            }
        }, {
            ...options,
            syncDirection: _constants.SYNC_DIRECTIONS.DO_NOT_SYNC,
            validate: {
                mustHaveKnownScopes () {
                    if (// `as` necessary because of incomplete typings in sequelize
                    this.scopes.some((scope)=>!Object.values(_constants.DEVICE_SCOPES).includes(scope))) {
                        throw new _errors.InvalidOperationError('Device must only use known scopes.');
                    }
                }
            }
        });
    }
    static initRelations(models) {
        this.belongsTo(models.User, {
            foreignKey: 'registeredById',
            as: 'registeredBy'
        });
    }
    static async getQuotaByUserId(userId) {
        return this.count({
            where: {
                registeredById: userId,
                [_sequelize.Op.or]: _constants.DEVICE_SCOPES_SUBJECT_TO_QUOTA.map((scope)=>// the jsonb operator `?`: "Does the string exist as a top-level key within the JSON value?"
                    _sequelize.Sequelize.literal(`scopes ? '${scope}'`))
            }
        });
    }
    async markSeen() {
        await this.update({
            lastSeenAt: _sequelize.Sequelize.fn('now')
        });
    }
    requiresQuota() {
        return this.scopes.some((scope)=>_constants.DEVICE_SCOPES_SUBJECT_TO_QUOTA.includes(scope));
    }
    ensureHasScope(scope) {
        if (!this.scopes.includes(scope)) {
            throw new Error('Device must have the required scope.');
        }
    }
};

//# sourceMappingURL=Device.js.map