"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
function _export(target, all) {
    for(var name in all)Object.defineProperty(target, name, {
        enumerable: true,
        get: all[name]
    });
}
_export(exports, {
    getIsLive: function() {
        return getIsLive;
    },
    getValues: function() {
        return getValues;
    },
    shouldForceRematerialise: function() {
        return shouldForceRematerialise;
    }
});
const _config = /*#__PURE__*/ _interop_require_default(require("config"));
const _constants = require("@tamanu/constants");
const _notes = require("@tamanu/shared/utils/notes");
const _fhirTypes = require("@tamanu/shared/services/fhirTypes");
const _fhir = require("@tamanu/shared/utils/fhir");
function _interop_require_default(obj) {
    return obj && obj.__esModule ? obj : {
        default: obj
    };
}
async function getValues(upstream, models) {
    const { ImagingRequest, LabRequest } = models;
    if (upstream instanceof ImagingRequest) return getValuesFromImagingRequest(upstream, models);
    if (upstream instanceof LabRequest) return getValuesFromLabRequest(upstream, models);
    throw new Error(`Invalid upstream type for service request ${upstream.constructor.name}`);
}
async function getValuesFromImagingRequest(upstream, models) {
    const { ImagingAreaExternalCode } = models;
    const areaExtCodes = new Map((await ImagingAreaExternalCode.findAll({
        where: {
            areaId: upstream.areas.map((area)=>area.id)
        }
    })).map((ext)=>[
            ext.areaId,
            {
                code: ext.code,
                description: ext.description,
                updatedAt: ext.updatedAt
            }
        ]));
    const subject = await _fhirTypes.FhirReference.to(models.FhirPatient, upstream.encounter?.patient?.id, {
        display: `${upstream.encounter?.patient?.firstName} ${upstream.encounter?.patient?.lastName}`
    });
    const encounter = await _fhirTypes.FhirReference.to(models.FhirEncounter, upstream.encounter?.id);
    const requester = await _fhirTypes.FhirReference.to(models.FhirPractitioner, upstream.requestedBy?.id, {
        display: upstream.requestedBy?.displayName
    });
    return {
        lastUpdated: new Date(),
        identifier: [
            new _fhirTypes.FhirIdentifier({
                system: _config.default.hl7.dataDictionaries.serviceRequestImagingId,
                value: upstream.id
            }),
            new _fhirTypes.FhirIdentifier({
                system: _config.default.hl7.dataDictionaries.serviceRequestImagingDisplayId,
                value: upstream.displayId
            })
        ],
        status: statusFromImagingRequest(upstream),
        intent: _constants.FHIR_REQUEST_INTENT.ORDER._,
        category: [
            new _fhirTypes.FhirCodeableConcept({
                coding: [
                    new _fhirTypes.FhirCoding({
                        system: 'http://snomed.info/sct',
                        code: '363679005'
                    })
                ]
            })
        ],
        priority: validatePriority(upstream.priority),
        code: imagingCode(upstream),
        orderDetail: upstream.areas.flatMap(({ id })=>areaExtCodes.has(id) ? [
                new _fhirTypes.FhirCodeableConcept({
                    text: areaExtCodes.get(id)?.description,
                    coding: [
                        new _fhirTypes.FhirCoding({
                            code: areaExtCodes.get(id)?.code,
                            system: _config.default.hl7.dataDictionaries.areaExternalCode
                        })
                    ]
                })
            ] : []),
        subject,
        encounter,
        occurrenceDateTime: (0, _fhir.formatFhirDate)(upstream.requestedDate),
        requester,
        locationCode: locationCode(upstream),
        note: imagingAnnotations(upstream),
        resolved: subject.isResolved() && encounter.isResolved() && requester.isResolved()
    };
}
async function getValuesFromLabRequest(upstream, models) {
    const subject = await _fhirTypes.FhirReference.to(models.FhirPatient, upstream.encounter?.patient?.id, {
        display: `${upstream.encounter?.patient?.firstName} ${upstream.encounter?.patient?.lastName}`
    });
    const encounter = await _fhirTypes.FhirReference.to(models.FhirEncounter, upstream.encounter?.id);
    const requester = await _fhirTypes.FhirReference.to(models.FhirPractitioner, upstream.requestedBy?.id, {
        display: upstream.requestedBy?.displayName
    });
    const specimen = await resolveSpecimen(upstream, models);
    return {
        lastUpdated: new Date(),
        identifier: [
            new _fhirTypes.FhirIdentifier({
                system: _config.default.hl7.dataDictionaries.serviceRequestLabId,
                value: upstream.id
            }),
            new _fhirTypes.FhirIdentifier({
                system: _config.default.hl7.dataDictionaries.serviceRequestLabDisplayId,
                value: upstream.displayId
            })
        ],
        status: statusFromLabRequest(upstream),
        intent: _constants.FHIR_REQUEST_INTENT.ORDER._,
        category: [
            new _fhirTypes.FhirCodeableConcept({
                coding: [
                    new _fhirTypes.FhirCoding({
                        system: 'http://snomed.info/sct',
                        code: '108252007'
                    })
                ]
            })
        ],
        priority: validatePriority(upstream.priority?.name),
        code: labCode(upstream),
        orderDetail: labOrderDetails(upstream),
        subject,
        encounter,
        occurrenceDateTime: (0, _fhir.formatFhirDate)(upstream.requestedDate),
        requester,
        note: labAnnotations(upstream),
        specimen: specimen ? [
            specimen
        ] : null,
        resolved: subject.isResolved() && encounter.isResolved() && requester.isResolved() && (specimen ? specimen.isResolved() : true)
    };
}
function resolveSpecimen(upstream, models) {
    if (!upstream.specimenAttached) {
        return null;
    }
    return _fhirTypes.FhirReference.to(models.FhirSpecimen, upstream.id);
}
function imagingCode(upstream) {
    const { imagingTypes } = _config.default.localisation.data;
    if (!imagingTypes) throw new _fhir.Exception('No imaging types specified in localisation.');
    const { imagingType } = upstream;
    const { label } = imagingTypes[imagingType] || {};
    if (!label) throw new _fhir.Exception(`No label matching imaging type ${imagingType} in localisation.`);
    return generateCodings(imagingType, undefined, label, _config.default.hl7.dataDictionaries.serviceRequestImagingTypeCodeSystem);
}
// Match the priority to a FHIR ServiceRequest priority where possible
// otherwise return null
// See: https://hl7.org/fhir/R4B/valueset-request-priority.html#expansion
function validatePriority(priority = '') {
    if (!Object.values(_constants.FHIR_REQUEST_PRIORITY).includes(priority)) {
        return null;
    }
    return priority;
}
function locationCode(upstream) {
    const facility = upstream.locationGroup?.facility ?? // most accurate
    upstream.location?.facility ?? // legacy data
    upstream.encounter?.location?.facility; // fallback to encounter
    if (!facility) return [];
    return [
        new _fhirTypes.FhirCodeableConcept({
            text: facility.name
        })
    ];
}
function statusFromImagingRequest(upstream) {
    return ({
        [_constants.IMAGING_REQUEST_STATUS_TYPES.CANCELLED]: _constants.FHIR_REQUEST_STATUS.REVOKED,
        [_constants.IMAGING_REQUEST_STATUS_TYPES.COMPLETED]: _constants.FHIR_REQUEST_STATUS.COMPLETED,
        [_constants.IMAGING_REQUEST_STATUS_TYPES.DELETED]: _constants.FHIR_REQUEST_STATUS.REVOKED,
        [_constants.IMAGING_REQUEST_STATUS_TYPES.ENTERED_IN_ERROR]: _constants.FHIR_REQUEST_STATUS.ENTERED_IN_ERROR,
        [_constants.IMAGING_REQUEST_STATUS_TYPES.IN_PROGRESS]: _constants.FHIR_REQUEST_STATUS.ACTIVE,
        [_constants.IMAGING_REQUEST_STATUS_TYPES.PENDING]: _constants.FHIR_REQUEST_STATUS.DRAFT
    })[upstream.status] ?? _constants.FHIR_REQUEST_STATUS.UNKNOWN;
}
function statusFromLabRequest(upstream) {
    switch(upstream.status){
        case _constants.LAB_REQUEST_STATUSES.SAMPLE_NOT_COLLECTED:
        case _constants.LAB_REQUEST_STATUSES.RECEPTION_PENDING:
            return _constants.FHIR_REQUEST_STATUS.DRAFT;
        case _constants.LAB_REQUEST_STATUSES.RESULTS_PENDING:
        case _constants.LAB_REQUEST_STATUSES.INTERIM_RESULTS:
        case _constants.LAB_REQUEST_STATUSES.TO_BE_VERIFIED:
        case _constants.LAB_REQUEST_STATUSES.VERIFIED:
            return _constants.FHIR_REQUEST_STATUS.ACTIVE;
        case _constants.LAB_REQUEST_STATUSES.PUBLISHED:
            return _constants.FHIR_REQUEST_STATUS.COMPLETED;
        case _constants.LAB_REQUEST_STATUSES.CANCELLED:
        case _constants.LAB_REQUEST_STATUSES.INVALIDATED:
        case _constants.LAB_REQUEST_STATUSES.DELETED:
            return _constants.FHIR_REQUEST_STATUS.REVOKED;
        case _constants.LAB_REQUEST_STATUSES.ENTERED_IN_ERROR:
            return _constants.FHIR_REQUEST_STATUS.ENTERED_IN_ERROR;
        default:
            return _constants.FHIR_REQUEST_STATUS.UNKNOWN;
    }
}
function getIsLive(upstream, models) {
    const { ImagingRequest, LabRequest } = models;
    if (upstream instanceof ImagingRequest) return _constants.IMAGING_TABLE_STATUS_GROUPINGS.ACTIVE.includes(upstream.status);
    if (upstream instanceof LabRequest) return _constants.LAB_REQUEST_TABLE_STATUS_GROUPINGS.ACTIVE.includes(upstream.status);
    throw new Error(`Invalid upstream type for service request ${upstream.constructor.name}`);
}
function shouldForceRematerialise(upstream, downstream, models) {
    const { ImagingRequest, LabRequest } = models;
    // Force remateralisation on status change
    if (upstream instanceof ImagingRequest) {
        return statusFromImagingRequest(upstream) !== downstream.status;
    }
    if (upstream instanceof LabRequest) {
        return statusFromLabRequest(upstream) !== downstream.status;
    }
    throw new Error(`Invalid upstream type for service request ${upstream.constructor.name}`);
}
function labCode(upstream) {
    const { labTestPanelRequest } = upstream;
    // ServiceRequests may not have a panel
    if (!labTestPanelRequest) {
        return null;
    }
    const { externalCode, name, code } = labTestPanelRequest.labTestPanel || {};
    return generateCodings(code, externalCode, name, _config.default.hl7.dataDictionaries.serviceRequestLabPanelCodeSystem, _config.default.hl7.dataDictionaries.serviceRequestLabPanelExternalCodeSystem);
}
function labOrderDetails({ tests }) {
    if (tests.length) {
        return tests.map(({ labTestType })=>{
            if (!labTestType) throw new _fhir.Exception('Received a null test');
            const { externalCode, code, name } = labTestType;
            return generateCodings(code, externalCode, name, _config.default.hl7.dataDictionaries.serviceRequestLabTestCodeSystem, _config.default.hl7.dataDictionaries.serviceRequestLabTestExternalCodeSystem);
        });
    }
    return [];
}
function labAnnotations(upstream) {
    return upstream.notes.map((note)=>{
        return new _fhirTypes.FhirAnnotation({
            time: (0, _fhir.formatFhirDate)(note.date),
            text: note.content
        });
    });
}
function imagingAnnotations(upstream) {
    // See EPI-451: imaging requests can embed notes about the area to image
    return (0, _notes.getNotesWithType)(upstream.notes, _constants.NOTE_TYPES.OTHER).map((note)=>new _fhirTypes.FhirAnnotation({
            time: (0, _fhir.formatFhirDate)(note.date),
            text: note.content
        }));
}
function generateCodings(code, externalCode, name, codeSystem, externalCodeSystem) {
    const coding = [];
    if (code) {
        coding.push(new _fhirTypes.FhirCoding({
            system: codeSystem,
            code,
            display: name
        }));
    }
    // Sometimes externalCode will not exists but if it does include it
    if (externalCode) {
        coding.push(new _fhirTypes.FhirCoding({
            system: externalCodeSystem,
            code: externalCode,
            display: name
        }));
    }
    if (coding.length > 0) {
        return new _fhirTypes.FhirCodeableConcept({
            coding,
            text: name
        });
    }
    return null;
}

//# sourceMappingURL=getValues.js.map