"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "ensureDeviceRegistration", {
    enumerable: true,
    get: function() {
        return ensureDeviceRegistration;
    }
});
const _sequelize = require("sequelize");
const _constants = require("@tamanu/constants");
const _errors = require("@tamanu/errors");
const _lodash = require("lodash");
async function ensureDeviceRegistration({ models: { Device }, settings, user, deviceId, scopes }) {
    // There's two race conditions we seek to avoid with this transaction:
    //
    // 1. If the device logs in to the server twice simultaneously, we want
    //    to avoid two SyncDevice records being created. This is also mitigated
    //    by the UNIQUE constraint on the table.
    //
    // 2. If two unknown devices try to register at the same time, and the
    //    quota is reached on the first one, the second should not be able to.
    await Device.sequelize.transaction({
        isolationLevel: _sequelize.Sequelize.Transaction.ISOLATION_LEVELS.SERIALIZABLE
    }, async ()=>{
        if (!deviceId) {
            if (scopes.includes(_constants.DEVICE_SCOPES.SYNC_CLIENT)) {
                throw new _errors.MissingCredentialError('Device ID is required');
            } else {
                return;
            }
        }
        const syncDevice = await Device.findByPk(deviceId);
        if (syncDevice) {
            if ((0, _lodash.difference)(scopes, syncDevice.scopes).length > 0) {
                throw new _errors.AuthPermissionError('Requested more scopes than the device has');
            }
            await syncDevice.markSeen();
            return;
        }
        const device = new Device({
            id: deviceId,
            registeredById: user.id,
            scopes
        });
        const deviceRegistrationQuotaEnabled = await settings.get(_constants.SETTING_KEYS.FEATURES_DEVICE_REGISTRATION_QUOTA_ENABLED);
        if (deviceRegistrationQuotaEnabled && device.requiresQuota()) {
            const currentCount = await Device.getQuotaByUserId(user.id);
            if (currentCount + 1 > user.deviceRegistrationQuota) {
                throw new _errors.QuotaExceededError();
            }
        }
        await device.save();
    });
}

//# sourceMappingURL=ensureDeviceRegistration.js.map