"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
function _export(target, all) {
    for(var name in all)Object.defineProperty(target, name, {
        enumerable: true,
        get: all[name]
    });
}
_export(exports, {
    getFlattenMergedPatientReplaceLinks: function() {
        return getFlattenMergedPatientReplaceLinks;
    },
    getFlattenMergedPatientReplacecByLinks: function() {
        return getFlattenMergedPatientReplacecByLinks;
    },
    getPatientWhereClause: function() {
        return getPatientWhereClause;
    },
    patientToHL7Patient: function() {
        return patientToHL7Patient;
    },
    patientToHL7PatientList: function() {
        return patientToHL7PatientList;
    }
});
const _config = /*#__PURE__*/ _interop_require_default(require("config"));
const _datefns = require("date-fns");
const _sequelize = require("sequelize");
const _lodash = require("lodash");
const _constants = require("@tamanu/constants");
const _utils = require("./utils");
const _hl7Parameters = require("./hl7Parameters");
const _hl7PatientFields = require("./hl7PatientFields");
function _interop_require_default(obj) {
    return obj && obj.__esModule ? obj : {
        default: obj
    };
}
function patientName(patient, additional) {
    const official = {
        use: 'official',
        prefix: additional.title ? [
            additional.title
        ] : [],
        // lastName is not a mandatory field in Tamanu, but is in HL7
        // Some patients genuinely do not have last names, so, just send
        // a preconfigured string in this circumstance.
        family: patient.lastName || _config.default.hl7.nullLastNameValue,
        given: [
            patient.firstName,
            patient.middleName
        ].filter((x)=>x)
    };
    if (!patient.culturalName) {
        return [
            official
        ];
    }
    return [
        official,
        {
            use: 'nickname',
            text: patient.culturalName
        }
    ];
}
function patientIds(patient, additional) {
    return [
        {
            use: 'usual',
            value: patient.displayId,
            assigner: _config.default.hl7.assigners.patientDisplayId,
            system: _config.default.hl7.dataDictionaries.patientDisplayId
        },
        {
            use: 'secondary',
            assigner: _config.default.hl7.assigners.patientPassport,
            value: additional.passportNumber
        },
        {
            use: 'secondary',
            assigner: _config.default.hl7.assigners.patientDrivingLicense,
            value: additional.drivingLicense
        }
    ].filter((x)=>x.value);
}
function patientAddress(patient, additional) {
    const { cityTown, streetVillage } = additional;
    if (!cityTown && !streetVillage) return [];
    return [
        {
            type: 'physical',
            use: 'home',
            city: additional.cityTown,
            line: additional.streetVillage ? [
                additional.streetVillage
            ] : []
        }
    ];
}
function patientTelecom(patient, additional) {
    return [
        additional.primaryContactNumber,
        additional.secondaryContactNumber
    ].filter((x)=>x).map((value, index)=>({
            rank: index + 1,
            value
        }));
}
function isPatientActive(patient) {
    if (patient.visibilityStatus === _constants.VISIBILITY_STATUSES.CURRENT) return true;
    return false;
}
const convertPatientToHL7Patient = (patient, additional = {})=>({
        resourceType: 'Patient',
        id: patient.id,
        active: isPatientActive(patient),
        identifier: patientIds(patient, additional),
        name: patientName(patient, additional),
        birthDate: patient.dateOfBirth,
        gender: patient.sex,
        address: patientAddress(patient, additional),
        telecom: patientTelecom(patient, additional),
        // Only add deceasedDateTime key if the patient is deceased
        ...patient.dateOfDeath && {
            deceasedDateTime: (0, _datefns.format)(new Date(patient.dateOfDeath), "yyyy-MM-dd'T'HH:mm:ssXXX")
        }
    });
function getFlattenMergedPatientReplaceLinks(baseUrl, patient, mergedPatientsByMergedIntoId, isRootPatientActive) {
    let links = [];
    const mergedPatients = mergedPatientsByMergedIntoId[patient.id] || [];
    for (const mergedPatient of mergedPatients){
        links.push({
            other: {
                reference: (0, _utils.getHL7Link)(`${baseUrl}/Patient/${mergedPatient.id}`)
            },
            type: isRootPatientActive ? _constants.FHIR_PATIENT_LINK_TYPES.REPLACES : _constants.FHIR_PATIENT_LINK_TYPES.SEE_ALSO
        });
        // get deeper level of merged patients if there's any
        links = links.concat(getFlattenMergedPatientReplaceLinks(baseUrl, mergedPatient, mergedPatientsByMergedIntoId, isRootPatientActive));
    }
    return links;
}
function getFlattenMergedPatientReplacecByLinks(baseUrl, patient, patientById) {
    let links = [];
    const supersededPatient = patientById[patient.mergedIntoId];
    if (patient.mergedIntoId && supersededPatient?.mergedIntoId) {
        links.push({
            other: {
                reference: (0, _utils.getHL7Link)(`${baseUrl}/Patient/${patient.mergedIntoId}`)
            },
            type: _constants.FHIR_PATIENT_LINK_TYPES.SEE_ALSO
        });
        links = links.concat(getFlattenMergedPatientReplacecByLinks(baseUrl, supersededPatient, patientById));
    } else {
        links.push({
            other: {
                reference: (0, _utils.getHL7Link)(`${baseUrl}/Patient/${patient.mergedIntoId}`)
            },
            type: _constants.FHIR_PATIENT_LINK_TYPES.REPLACED_BY
        });
    }
    return links;
}
const getMergedPatientByMergedIntoId = async (models)=>{
    const allMergedPatients = await models.Patient.findAll({
        attributes: [
            'id',
            'mergedIntoId'
        ],
        where: {
            mergedIntoId: {
                [_sequelize.Op.not]: null
            }
        },
        paranoid: false
    });
    return (0, _lodash.groupBy)(allMergedPatients, 'mergedIntoId');
};
const getPatientById = async (models)=>{
    // This is going to fetch all active and inactive patients
    // which can be memory heavy. But we only fetch the minimum
    // info that we need which is id and mergedIntoId.
    // So I think it is acceptable.
    const patients = await models.Patient.findAll({
        attributes: [
            'id',
            'mergedIntoId'
        ],
        paranoid: false,
        where: {
            visibilityStatus: {
                [_sequelize.Op.in]: [
                    _constants.VISIBILITY_STATUSES.CURRENT,
                    _constants.VISIBILITY_STATUSES.MERGED
                ]
            }
        }
    });
    return (0, _lodash.keyBy)(patients, 'id');
};
const getPatientLinks = (baseUrl, patient, patientById, mergedPatientsByMergedIntoId, isRootPatientActive)=>{
    let links = [];
    // Get 'replaced-by/seealso' links of a patient if there's any
    if (patient.mergedIntoId) {
        links = getFlattenMergedPatientReplacecByLinks(baseUrl, patient, patientById);
    }
    const mergedPatients = mergedPatientsByMergedIntoId[patient.id] || [];
    // Get 'replaces' links of a patient if there's any
    if (mergedPatients.length) {
        links = links.concat(getFlattenMergedPatientReplaceLinks(baseUrl, patient, mergedPatientsByMergedIntoId, isRootPatientActive));
    }
    // Reorder seealso links to be after replaces/replaced-by links
    const seeAlsoLinks = links.filter((l)=>l.type === _constants.FHIR_PATIENT_LINK_TYPES.SEE_ALSO);
    const otherLinks = links.filter((l)=>l.type !== _constants.FHIR_PATIENT_LINK_TYPES.SEE_ALSO);
    return [
        ...otherLinks,
        ...seeAlsoLinks
    ];
};
const patientToHL7Patient = async (req, patient, additional = {})=>{
    const { models } = req.store;
    const baseUrl = (0, _utils.getBaseUrl)(req, false);
    const mergedPatientsByMergedIntoId = await getMergedPatientByMergedIntoId(models);
    const patientById = await getPatientById(models);
    const isRootPatientActive = Boolean(!patient.mergedIntoId);
    const links = getPatientLinks(baseUrl, patient, patientById, mergedPatientsByMergedIntoId, isRootPatientActive);
    const hl7Patient = convertPatientToHL7Patient(patient, additional);
    return {
        ...hl7Patient,
        ...links.length ? {
            link: links
        } : {}
    };
};
const patientToHL7PatientList = async (req, patients)=>{
    const { models } = req.store;
    const baseUrl = (0, _utils.getBaseUrl)(req, false);
    const mergedPatientsByMergedIntoId = await getMergedPatientByMergedIntoId(models);
    const patientById = await getPatientById(models);
    const hl7Patients = [];
    for (const patient of patients){
        const hl7Patient = convertPatientToHL7Patient(patient, patient.additionalData?.[0]);
        const isRootPatientActive = Boolean(!patient.mergedIntoId);
        const links = getPatientLinks(baseUrl, patient, patientById, mergedPatientsByMergedIntoId, isRootPatientActive);
        hl7Patients.push({
            ...hl7Patient,
            ...links.length ? {
                link: links
            } : {}
        });
    }
    return hl7Patients;
};
function getPatientWhereClause(displayId, query = {}) {
    // to only get active or merged patients
    const filters = [
        {
            [_sequelize.Op.or]: [
                {
                    visibilityStatus: _constants.VISIBILITY_STATUSES.CURRENT,
                    deletedAt: null
                },
                {
                    visibilityStatus: _constants.VISIBILITY_STATUSES.MERGED
                }
            ]
        }
    ];
    // Handle search by ID separately
    if (displayId) {
        filters.push({
            displayId
        });
    }
    // Create a filter for each query param
    Object.entries(query).forEach(([key, value])=>{
        const [parameter, modifier] = (0, _utils.getParamAndModifier)(key);
        // Only create filter if the parameter is a supported patient field
        if (parameter in _hl7PatientFields.hl7PatientFields === false) {
            return;
        }
        const { fieldName, columnName, parameterType, getValue, getOperator } = _hl7PatientFields.hl7PatientFields[parameter];
        const defaultOperator = getOperator ? getOperator(value) : (0, _utils.getDefaultOperator)(parameterType);
        const operator = modifier ? _hl7Parameters.modifiers[parameterType][modifier] : defaultOperator;
        const extractedValue = getValue ? getValue(value) : value;
        const queryObject = (0, _utils.getQueryObject)(columnName, extractedValue, operator, modifier, parameterType);
        filters.push({
            [fieldName]: queryObject
        });
    });
    // Wrap all filters with explicit "AND" if they exist,
    // otherwise return empty object
    return filters.length > 0 ? {
        [_sequelize.Op.and]: filters
    } : {};
}

//# sourceMappingURL=patient.js.map