"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "createEuDccVaccinationData", {
    enumerable: true,
    get: function() {
        return createEuDccVaccinationData;
    }
});
const _datefnstz = require("date-fns-tz");
const _transliteration = require("transliteration");
const _config = /*#__PURE__*/ _interop_require_default(require("config"));
const _constants = require("@tamanu/constants");
const _uvci = require("@tamanu/shared/utils/uvci");
const _localisation = require("../../localisation");
function _interop_require_default(obj) {
    return obj && obj.__esModule ? obj : {
        default: obj
    };
}
const SCHEDULE_TO_SEQUENCE = {
    'Dose 1': 1,
    'Dose 2': 2,
    Booster: 3,
    'Dose 3': 3,
    'Dose 4': 4,
    'Dose 5': 5,
    'Dose 6': 6,
    'Dose 7': 7,
    'Dose 8': 8,
    'Dose 9': 9
};
const FORMAT_ISODATE = 'yyyy-MM-dd';
async function createEuDccVaccinationData(administeredVaccineId, { models }) {
    const { Patient, ReferenceData, AdministeredVaccine, Encounter, Facility, Location, ScheduledVaccine, CertifiableVaccine } = models;
    const vaccination = await AdministeredVaccine.findByPk(administeredVaccineId, {
        include: [
            {
                model: Location,
                as: 'location',
                include: [
                    {
                        model: Facility,
                        as: 'facility'
                    }
                ]
            },
            {
                model: Encounter,
                as: 'encounter',
                include: [
                    {
                        model: Patient,
                        as: 'patient'
                    },
                    {
                        model: Location,
                        as: 'location',
                        include: [
                            {
                                model: Facility,
                                as: 'facility'
                            }
                        ]
                    }
                ]
            },
            {
                model: ScheduledVaccine,
                as: 'scheduledVaccine',
                include: [
                    {
                        model: ReferenceData,
                        as: 'vaccine'
                    }
                ]
            }
        ]
    });
    if (!vaccination) {
        throw new Error(`No vaccination found with id=${administeredVaccineId}`);
    }
    if (vaccination.status !== 'GIVEN') {
        throw new Error('Vaccination is not given');
    }
    const { id, date, location, scheduledVaccine: { doseLabel, vaccine: { id: vaccineId } }, encounter: { patient, location: { facility: { name: encounterFacilityName } } } } = vaccination;
    const facilityName = location?.facility?.name ?? encounterFacilityName;
    const certVax = await CertifiableVaccine.findOne({
        where: {
            vaccineId
        },
        include: [
            {
                model: ReferenceData,
                as: 'manufacturer'
            }
        ]
    });
    if (!certVax) throw new Error('Vaccine is not certifiable');
    if (!certVax.usableForEuDcc()) throw new Error('Vaccination is not usable for EU DCC');
    const { timeZone, country } = await (0, _localisation.getLocalisation)();
    const dob = (0, _datefnstz.formatInTimeZone)(patient.dateOfBirth, timeZone, FORMAT_ISODATE);
    const vaxDate = (0, _datefnstz.formatInTimeZone)(date, timeZone, FORMAT_ISODATE);
    return {
        ver: _constants.EUDCC_SCHEMA_VERSION,
        nam: nameSection(patient),
        dob,
        [_constants.EUDCC_CERTIFICATE_TYPES.VACCINATION]: [
            {
                tg: certVax.targetCode,
                vp: certVax.vaccineCode,
                mp: certVax.euProductCode,
                ma: certVax.manufacturer.code,
                dn: SCHEDULE_TO_SEQUENCE[doseLabel],
                sd: certVax.maximumDosage,
                dt: vaxDate,
                co: country['alpha-2'],
                is: _config.default.integrations.euDcc.issuer ?? facilityName,
                ci: (0, _uvci.generateUVCI)(id, {
                    format: 'eudcc',
                    countryCode: country['alpha-2']
                })
            }
        ]
    };
}
function transliterate(name) {
    return (0, _transliteration.transliterate)(name.toUpperCase()) // transliterate to ASCII
    .replace("'", '') // apostrophes shall be omitted
    .replace('-', '<') // hyphens as single filler
    .replace(/(\s+|,\s*)/g, '<') // commas as single filler (name parts are separated here)
    .replace(/<+/g, '<') // collapse multiple fillers
    .replace(/[^A-Z<]+/g, '') // all punctuation shall be omitted
    .substring(0, 80); // maximum length is 80
}
function nameSection(patient) {
    const { firstName, lastName } = patient;
    return {
        fn: lastName,
        fnt: transliterate(lastName),
        gn: firstName,
        gnt: transliterate(firstName)
    };
}

//# sourceMappingURL=Translation.js.map