"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "dataGenerator", {
    enumerable: true,
    get: function() {
        return dataGenerator;
    }
});
const _lodash = require("lodash");
const _datefns = require("date-fns");
const _sequelize = require("sequelize");
const _utilities = require("../utilities");
const _utils = require("./utils");
const _dateTime = require("@tamanu/utils/dateTime");
const _constants = require("./constants");
const getReferralByResponseIds = async (models, surveyResponseIds)=>models.Referral.findAll({
        where: {
            surveyResponseId: {
                [_sequelize.Op.in]: surveyResponseIds
            }
        },
        include: [
            {
                model: models.Encounter,
                as: 'initiatingEncounter'
            },
            {
                model: models.SurveyResponse,
                as: 'surveyResponse'
            }
        ]
    });
const getReferralByPatientSurveyAndDate = async (models, transformedAnswers)=>{
    const responseIds = (0, _lodash.uniqWith)(transformedAnswers.map((a)=>a.surveyResponseId), _lodash.isEqual);
    const referrals = await getReferralByResponseIds(models, responseIds);
    const finalReferrals = await (0, _utils.removeDuplicatedReferralsPerDate)(referrals);
    return (0, _lodash.keyBy)(finalReferrals, (r)=>{
        const referralDate = (0, _dateTime.format)(r.surveyResponse.endTime, 'dd-MM-yyyy');
        const surveyGroupKey = (0, _utils.getSurveyGroupKey)(r.surveyResponse.surveyId);
        return `${r.initiatingEncounter.patientId}|${surveyGroupKey}|${referralDate}`;
    });
};
const getPerPatientPerSurveyPerDateKey = (patientId, surveyGroupKey, date)=>`${patientId}|${surveyGroupKey}|${date}`;
const dataGenerator = async ({ models }, parameters = {})=>{
    const answerWhereClause = (0, _utils.parametersToAnswerSqlWhere)(parameters);
    const rawAnswers = await (0, _utilities.getAnswers)(models, answerWhereClause);
    const filteredAnswers = await (0, _utils.transformAndRemoveDuplicatedAnswersPerDate)(models, rawAnswers, _constants.ALL_SURVEY_IDS);
    const referralByPatientSurveyAndDate = await getReferralByPatientSurveyAndDate(models, filteredAnswers);
    const patientById = await (0, _utils.getPatientById)(models, rawAnswers);
    const answersByPatientId = (0, _lodash.groupBy)(filteredAnswers, (a)=>a.patientId);
    const answersByPatientSurveyDataElement = (0, _lodash.keyBy)(filteredAnswers, (a)=>{
        const responseDate = (0, _dateTime.format)(a.responseEndTime, 'dd-MM-yyyy');
        const surveyGroupKey = (0, _utils.getSurveyGroupKey)(a.surveyId);
        return (0, _utils.getPerPatientPerSurveyPerDatePerElementKey)(a.patientId, surveyGroupKey, responseDate, a.dataElementId);
    });
    const reportData = [];
    // Report should create a new line for each patient each time they have one of the below screening forms submitted:
    // CVD Primary Screening Form
    // Breast Cancer Primary Screening Form
    // Cervical Cancer Primary Screening Form
    // If there are multiple survey/referral submission on the same date, pull the latest answer for every data element regardless of which survey response
    // Referral details should be pulled into the report if they are submitted on the same day as the corresponding screening survey
    // Group the records by patient
    for (const [patientId, patientAnswers] of Object.entries(answersByPatientId)){
        const screeningFormAnswers = patientAnswers.filter((a)=>_constants.FORM_SURVEY_IDS.includes(a.surveyId));
        const groupedScreeningFormAnswers = (0, _lodash.groupBy)(screeningFormAnswers, (a)=>{
            const responseDate = (0, _dateTime.format)(a.responseEndTime, 'dd-MM-yyyy');
            return `${(0, _utils.getSurveyGroupKey)(a.surveyId)}|${responseDate}`;
        });
        const patient = patientById[patientId];
        const patientAdditionalData = patient.additionalData?.[0];
        // Group the answers by survey and date. So for per patient per date, we should 1 row per survey (maximum 3 surveys)
        for (const [key] of Object.entries(groupedScreeningFormAnswers)){
            const [surveyGroupKey, responseDate] = key.split('|');
            const age = patient.dateOfBirth ? (0, _dateTime.ageInYears)(patient.dateOfBirth) : '';
            const recordData = {
                firstName: patient.firstName,
                lastName: patient.lastName,
                displayId: patient.displayId,
                age,
                gender: patient.sex,
                ethnicity: patientAdditionalData?.ethnicity?.name,
                contactNumber: patientAdditionalData?.primaryContactNumber,
                village: patient.village?.name,
                medicalArea: patientAdditionalData?.medicalArea?.name,
                nursingZone: patientAdditionalData?.nursingZone?.name,
                screeningCompleted: _constants.FORM_NAME_BY_SURVEY_GROUP_KEY[surveyGroupKey]
            };
            const formDataElements = (0, _utils.getFormDataElements)(surveyGroupKey);
            Object.entries(formDataElements).forEach(([dataKey, dataElementId])=>{
                recordData[dataKey] = (0, _utils.getCachedAnswer)(answersByPatientSurveyDataElement, patientId, surveyGroupKey, responseDate, dataElementId);
            });
            const referral = referralByPatientSurveyAndDate[getPerPatientPerSurveyPerDateKey(patientId, surveyGroupKey, responseDate)];
            recordData.referralCreated = referral ? 'Yes' : 'No';
            // If referral has been created on the same date, populate the referral details
            if (referral) {
                recordData.referralStatus = (0, _lodash.upperFirst)(referral.status);
                const referralDataElements = (0, _utils.getReferralDataElements)(surveyGroupKey);
                Object.entries(referralDataElements).forEach(([dataKey, dataElementId])=>{
                    recordData[dataKey] = (0, _utils.getCachedAnswer)(answersByPatientSurveyDataElement, patientId, surveyGroupKey, responseDate, dataElementId);
                });
            }
            reportData.push(recordData);
        }
    }
    const sortedReportData = reportData.sort(({ dateOfScreening: date1 }, { dateOfScreening: date2 })=>{
        if (date2 && !date1) return 1;
        if (date1 && !date2) return -1;
        if (!date1 && !date2) return 0;
        // Sort oldest to most recent
        return (0, _datefns.parseISO)(date1) - (0, _datefns.parseISO)(date2);
    });
    return (0, _utilities.generateReportFromQueryData)(sortedReportData, _constants.PRIMARY_SCREENING_REPORT_COLUMN_TEMPLATE);
};

//# sourceMappingURL=fiji-ncd-primary-screening-line-list.js.map