"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "invoices", {
    enumerable: true,
    get: function() {
        return invoiceRoute;
    }
});
const _express = /*#__PURE__*/ _interop_require_default(require("express"));
const _expressasynchandler = /*#__PURE__*/ _interop_require_default(require("express-async-handler"));
const _nanoid = require("nanoid");
const _errors = require("@tamanu/errors");
const _constants = require("@tamanu/constants");
const _uuid = require("uuid");
const _zod = require("zod");
const _sequelize = require("sequelize");
const _invoiceItems = require("./invoiceItems");
const _countryDateTime = require("@tamanu/shared/utils/countryDateTime");
const _patientPayment = require("./patientPayment");
const _lodash = require("lodash");
function _interop_require_default(obj) {
    return obj && obj.__esModule ? obj : {
        default: obj
    };
}
const invoiceNumberGenerator = (0, _nanoid.customAlphabet)('123456789ABCDEFGHIJKLMNPQRSTUVWXYZ', 10);
const invoiceRoute = _express.default.Router();
//* Create invoice
const createInvoiceSchema = _zod.z.object({
    encounterId: _zod.z.string().uuid(),
    discount: _zod.z.object({
        percentage: _zod.z.coerce.number().min(0).max(1).transform((amount)=>(0, _lodash.round)(amount, 2)),
        reason: _zod.z.string().optional(),
        isManual: _zod.z.boolean()
    }).strip().optional(),
    date: _zod.z.string()
}).strip().transform((data)=>({
        ...data,
        id: (0, _uuid.v4)(),
        displayId: invoiceNumberGenerator(),
        status: _constants.INVOICE_STATUSES.IN_PROGRESS
    }));
invoiceRoute.post('/', (0, _expressasynchandler.default)(async (req, res)=>{
    req.checkPermission('create', 'Invoice');
    const { body: { facilityId, ...body } } = req;
    const { data, error } = await createInvoiceSchema.safeParseAsync(body);
    if (error) throw new _errors.ValidationError(error.message);
    // get encounter
    const encounter = await req.models.Encounter.findByPk(data.encounterId, {
        attributes: [
            'patientId'
        ]
    });
    if (!encounter) throw new _errors.ValidationError(`encounter ${data.encounterId} not found`);
    const insurerId = await req.models.PatientAdditionalData.findOne({
        where: {
            patientId: encounter.patientId
        },
        attributes: [
            'insurerId'
        ]
    }).then((patientData)=>patientData?.insurerId);
    const insurerPercentage = await req.settings[facilityId].get(_constants.SETTING_KEYS.INSURER_DEFAUlT_CONTRIBUTION);
    const defaultInsurer = insurerId && insurerPercentage ? {
        insurerId,
        percentage: insurerPercentage
    } : null;
    // create invoice transaction
    const transaction = await req.db.transaction();
    try {
        //create invoice
        const invoice = await req.models.Invoice.create(data, {
            transaction
        });
        // insert default insurer
        if (defaultInsurer) await req.models.InvoiceInsurer.create({
            invoiceId: data.id,
            ...defaultInsurer
        }, {
            transaction
        });
        // create invoice discount
        if (data.discount) await req.models.InvoiceDiscount.create({
            ...data.discount,
            invoiceId: data.id,
            appliedByUserId: req.user.id,
            appliedTime: (0, _countryDateTime.getCurrentCountryTimeZoneDateTimeString)()
        }, {
            transaction
        });
        await transaction.commit();
        res.json(invoice);
    } catch (error) {
        await transaction.rollback();
        throw error;
    }
}));
//* Update invoice
const updateInvoiceSchema = _zod.z.object({
    discount: _zod.z.object({
        id: _zod.z.string().uuid().default(_uuid.v4),
        percentage: _zod.z.coerce.number().min(0).max(1).transform((amount)=>(0, _lodash.round)(amount, 2)),
        reason: _zod.z.string().optional(),
        isManual: _zod.z.boolean()
    }).strip().optional(),
    insurers: _zod.z.object({
        id: _zod.z.string().uuid().default(_uuid.v4),
        percentage: _zod.z.coerce.number().min(0).max(1).transform((amount)=>(0, _lodash.round)(amount, 2)),
        insurerId: _zod.z.string()
    }).strip().array().refine((insurers)=>insurers.reduce((sum, insurer)=>sum += insurer.percentage, 0) <= 1, 'Total insurer percentage should not exceed 100%'),
    items: _zod.z.object({
        id: _zod.z.string().uuid().default(_uuid.v4),
        orderDate: _zod.z.string().date(),
        orderedByUserId: _zod.z.string(),
        productId: _zod.z.string(),
        productName: _zod.z.string(),
        productPrice: _zod.z.coerce.number().transform((amount)=>(0, _lodash.round)(amount, 2)),
        productCode: _zod.z.string().default(''),
        productDiscountable: _zod.z.boolean().default(true),
        quantity: _zod.z.coerce.number().default(1),
        note: _zod.z.string().optional(),
        sourceId: _zod.z.string().uuid().optional(),
        discount: _zod.z.object({
            id: _zod.z.string().uuid().default(_uuid.v4),
            type: _zod.z.enum(Object.values(_constants.INVOICE_ITEMS_DISCOUNT_TYPES)),
            amount: _zod.z.coerce.number().transform((amount)=>(0, _lodash.round)(amount, 2)),
            reason: _zod.z.string().optional()
        }).strip().optional()
    }).strip().refine((item)=>{
        if (!item.discount) return true;
        if (item.discount.type === _constants.INVOICE_ITEMS_DISCOUNT_TYPES.PERCENTAGE) {
            return item.discount.amount < 0 ? true : item.discount.amount <= 1 && item.discount.amount > 0;
        }
        return item.discount.amount <= item.productPrice * item.quantity;
    }, 'Invalid discount amount').array()
}).strip();
/**
 * Update invoice
 * - Only in progress invoices can be updated
 */ invoiceRoute.put('/:id/', (0, _expressasynchandler.default)(async (req, res)=>{
    req.checkPermission('write', 'Invoice');
    const invoiceId = req.params.id;
    const foundInvoice = await req.models.Invoice.findByPk(invoiceId);
    if (!foundInvoice) throw new _errors.NotFoundError(`Unable to find invoice ${invoiceId}`);
    //* Only in progress invoices can be updated
    if (foundInvoice.status !== _constants.INVOICE_STATUSES.IN_PROGRESS) throw new _errors.InvalidOperationError('Only in progress invoices can be updated');
    const { data, error } = await updateInvoiceSchema.safeParseAsync(req.body);
    if (error) throw new _errors.ValidationError(error.message);
    const transaction = await req.db.transaction();
    try {
        if (!data.discount) {
            //remove any existing discount if discount info is not provided
            await req.models.InvoiceDiscount.destroy({
                where: {
                    invoiceId
                }
            }, {
                transaction
            });
        } else {
            //remove any existing discount if discount id is not matching
            await req.models.InvoiceDiscount.destroy({
                where: {
                    invoiceId,
                    id: {
                        [_sequelize.Op.ne]: data.discount.id
                    }
                }
            }, {
                transaction
            });
            //update or create discount
            await req.models.InvoiceDiscount.upsert({
                ...data.discount,
                invoiceId,
                appliedByUserId: req.user.id,
                appliedTime: (0, _countryDateTime.getCurrentCountryTimeZoneDateTimeString)()
            }, {
                transaction
            });
        }
        //remove any existing insurer if insurer ids are not matching
        await req.models.InvoiceInsurer.destroy({
            where: {
                invoiceId,
                id: {
                    [_sequelize.Op.notIn]: data.insurers.map((insurer)=>insurer.id)
                }
            }
        }, {
            transaction
        });
        //update or create insurers
        for (const insurer of data.insurers){
            await req.models.InvoiceInsurer.upsert({
                ...insurer,
                invoiceId
            }, {
                transaction
            });
        }
        //remove any existing item if item ids are not matching
        await req.models.InvoiceItem.destroy({
            where: {
                invoiceId,
                id: {
                    [_sequelize.Op.notIn]: data.items.map((item)=>item.id)
                }
            }
        }, {
            transaction
        });
        for (const item of data.items){
            const { discount: itemDiscount, ...itemData } = item;
            //update or create item
            await req.models.InvoiceItem.upsert({
                ...itemData,
                invoiceId
            }, {
                transaction
            });
            //remove any existing discount if discount info is not provided
            if (!itemDiscount) {
                await req.models.InvoiceItemDiscount.destroy({
                    where: {
                        invoiceItemId: item.id
                    }
                }, {
                    transaction
                });
            } else {
                //remove any existing discount if discount id is not matching
                await req.models.InvoiceItemDiscount.destroy({
                    where: {
                        invoiceItemId: item.id,
                        id: {
                            [_sequelize.Op.ne]: itemDiscount.id
                        }
                    }
                }, {
                    transaction
                });
                //update or create discount
                await req.models.InvoiceItemDiscount.upsert({
                    ...itemDiscount,
                    invoiceItemId: item.id
                }, {
                    transaction
                });
            }
        }
        await transaction.commit();
    } catch (error) {
        await transaction.rollback();
        throw error;
    }
    const invoice = await req.models.Invoice.findByPk(invoiceId);
    res.json(invoice.dataValues);
}));
/**
 * Cancel invoice
 */ invoiceRoute.put('/:id/cancel', (0, _expressasynchandler.default)(async (req, res)=>{
    req.checkPermission('write', 'Invoice');
    const invoiceId = req.params.id;
    const invoice = await req.models.Invoice.findByPk(invoiceId, {
        attributes: [
            'id',
            'status'
        ]
    });
    if (!invoice) throw new _errors.NotFoundError('Invoice not found');
    //only in progress invoices can be cancelled
    if (invoice.status !== _constants.INVOICE_STATUSES.IN_PROGRESS) {
        throw new _errors.InvalidOperationError('Only in progress invoices can be cancelled');
    }
    invoice.status = _constants.INVOICE_STATUSES.CANCELLED;
    await invoice.save();
    res.send(invoice);
}));
/**
 * Finalize invoice
 * - An invoice cannot be finalised until the Encounter has been closed
 * - Only in progress invoices can be finalised
 * - Invoice items data will be frozen
 */ invoiceRoute.put('/:id/finalize', (0, _expressasynchandler.default)(async (req, res)=>{
    req.checkPermission('write', 'Invoice');
    const invoiceId = req.params.id;
    const invoice = await req.models.Invoice.findByPk(invoiceId, {
        attributes: [
            'id',
            'status'
        ]
    });
    if (!invoice) throw new _errors.NotFoundError('Invoice not found');
    //only in progress invoices can be finalised
    if (invoice.status !== _constants.INVOICE_STATUSES.IN_PROGRESS) {
        throw new _errors.InvalidOperationError('Only in progress invoices can be finalised');
    }
    //An invoice cannot be finalised until the Encounter has been closed
    //an encounter is considered closed if it has an end date
    const encounterClosed = await req.models.Encounter.findByPk(invoice.encounterId, {
        attributes: [
            'endDate'
        ]
    }).then((encounter)=>!!encounter?.endDate);
    if (encounterClosed) {
        throw new _errors.InvalidOperationError('Ivnvoice cannot be finalised until the Encounter has been closed');
    }
    invoice.status = _constants.INVOICE_STATUSES.FINALISED;
    await invoice.save();
    res.json(invoice);
}));
/**
 * Finalize invoice
 * You cannot delete a Finalised invoice
 * You can delete a cancelled or in progress invoice
 */ invoiceRoute.delete('/:id', (0, _expressasynchandler.default)(async (req, res)=>{
    req.checkPermission('delete', 'Invoice');
    const invoiceId = req.params.id;
    const invoice = await req.models.Invoice.findByPk(invoiceId, {
        attributes: [
            'id',
            'status'
        ]
    });
    if (!invoice) throw new _errors.NotFoundError('Invoice not found');
    //Finalised invoices cannot be deleted
    if (invoice.status === _constants.INVOICE_STATUSES.FINALISED) {
        throw new _errors.InvalidOperationError('Only in progress invoices can be finalised');
    }
    await invoice.destroy();
    res.status(204).send();
}));
invoiceRoute.use(_invoiceItems.invoiceItemsRoute);
invoiceRoute.use(_patientPayment.patientPaymentRoute);

//# sourceMappingURL=invoices.js.map