"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
function _export(target, all) {
    for(var name in all)Object.defineProperty(target, name, {
        enumerable: true,
        get: all[name]
    });
}
_export(exports, {
    getAgeDurationFromDate: function() {
        return getAgeDurationFromDate;
    },
    getDisplayAge: function() {
        return getDisplayAge;
    }
});
const _datefns = require("date-fns");
const _dateTime = require("./dateTime");
const getAgeDurationFromDate = (date)=>{
    const start = (0, _dateTime.parseDate)(date);
    if (!start) return null;
    return (0, _datefns.intervalToDuration)({
        start,
        end: new Date()
    });
};
//? Seems like these are not used anywhere
// NB: If you're trying to format a date as a string:
// - if you're storing it or communicating with it, you should keep it as a
//   native date object if possible
// - if you're displaying it to a user, you should use the DateDisplay component
//   instead, it'll help keep date rendering consistent throughout the app
// export const getAgeFromDate = (date: number | string | Date) => {
//   return differenceInYears(new Date(), new Date(date));
// };
const getDifferenceFnByUnit = {
    years: _datefns.differenceInYears,
    months: _datefns.differenceInMonths,
    weeks: _datefns.differenceInWeeks,
    days: _datefns.differenceInDays
};
// eslint-disable-next-line no-unused-vars
const comparators = {
    '>': (left, right)=>left > right,
    '<': (left, right)=>left < right,
    '>=': (left, right)=>left >= right,
    '<=': (left, right)=>left <= right
};
const compareDate = (leftDate, operator, rightDate, exclusive = false)=>{
    let comparator = operator;
    if (!exclusive) comparator += '=';
    const comparatorFn = comparators[comparator];
    return comparatorFn?.(leftDate, rightDate) ?? false;
};
const ageIsWithinRange = (birthDate, range)=>{
    const { duration: minDuration, exclusive: minExclusive } = range.min ?? {};
    const { duration: maxDuration, exclusive: maxExclusive } = range.max ?? {};
    const minDate = minDuration ? (0, _datefns.startOfDay)((0, _datefns.add)(birthDate, minDuration)) : -Infinity;
    const maxDate = maxDuration ? (0, _datefns.startOfDay)((0, _datefns.add)(birthDate, maxDuration)) : Infinity;
    const now = (0, _datefns.startOfDay)(new Date());
    return (!minDate || compareDate(minDate, '<', now, minExclusive)) && (!maxDate || compareDate(now, '<', maxDate, maxExclusive));
};
const getDisplayAge = (dateOfBirth, ageDisplayFormat)=>{
    if (!ageDisplayFormat || !dateOfBirth || !(0, _dateTime.isISOString)(dateOfBirth)) {
        return '';
    }
    const ageDuration = getAgeDurationFromDate(dateOfBirth);
    if (!ageDuration) return '';
    const birthDate = (0, _dateTime.parseDate)(dateOfBirth);
    if (!birthDate) return '';
    for (const displayFormat of ageDisplayFormat){
        const { as, range } = displayFormat;
        if (ageIsWithinRange(birthDate, range)) {
            const differenceFn = getDifferenceFnByUnit[as];
            const value = differenceFn(new Date(), birthDate);
            const unit = as.slice(0, -1); // slice off the s
            return `${value} ${unit}${value === 1 ? '' : 's'}`;
        }
    }
    return (0, _datefns.formatDuration)(ageDuration, {
        format: [
            'years'
        ]
    });
};

//# sourceMappingURL=date.js.map