"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "dataGenerator", {
    enumerable: true,
    get: function() {
        return dataGenerator;
    }
});
const _sequelize = require("sequelize");
const _datefns = require("date-fns");
const _constants = require("@tamanu/constants");
const _dateTime = require("@tamanu/utils/dateTime");
const _utilities = require("./utilities");
const FIELD_TO_TITLE = {
    firstName: 'First name',
    lastName: 'Last name',
    displayId: 'Patient ID',
    age: 'Age',
    sex: 'Gender',
    ethnicity: 'Ethnicity',
    contactPhone: 'Contact number',
    subdivision: 'Subdivision',
    medicalArea: 'Medical Area',
    nursingZone: 'Nursing Zone',
    clinician: 'Clinician',
    dateOfAttendance: 'Date of attendance',
    department: 'Department',
    locationGroup: 'Area',
    location: 'Location',
    reasonForAttendance: 'Reason for attendance',
    primaryDiagnosis: 'Primary diagnosis',
    otherDiagnoses: 'Other diagnoses'
};
const reportColumnTemplate = Object.entries(FIELD_TO_TITLE).map(([key, title])=>({
        title,
        accessor: (data)=>data[key]
    }));
const parametersToEncounterSqlWhere = (parameters)=>{
    const newParameters = {
        ...parameters
    };
    if (parameters.fromDate) {
        newParameters.fromDate = (0, _dateTime.toDateTimeString)((0, _datefns.startOfDay)((0, _datefns.parseISO)(parameters.fromDate)));
    }
    if (parameters.toDate) {
        newParameters.toDate = (0, _dateTime.toDateTimeString)((0, _datefns.endOfDay)((0, _datefns.parseISO)(parameters.toDate)));
    }
    return Object.entries(newParameters).filter(([, val])=>val).reduce((where, [key, value])=>{
        const newWhere = {
            ...where
        };
        switch(key){
            case 'village':
                newWhere['$patient.village_id$'] = value;
                break;
            case 'diagnosis':
                newWhere['$diagnoses.diagnosis_id$'] = value;
                break;
            case 'fromDate':
                if (!newWhere.startDate) {
                    newWhere.startDate = {};
                }
                newWhere.startDate[_sequelize.Op.gte] = value;
                break;
            case 'toDate':
                if (!newWhere.startDate) {
                    newWhere.startDate = {};
                }
                newWhere.startDate[_sequelize.Op.lte] = value;
                break;
            default:
                break;
        }
        return newWhere;
    }, {});
};
const getEncounters = async (models, parameters)=>{
    const encounters = await models.Encounter.findAll({
        attributes: [
            'startDate',
            'reasonForEncounter',
            'id'
        ],
        include: [
            {
                model: models.Patient,
                as: 'patient',
                attributes: [
                    'firstName',
                    'lastName',
                    'displayId',
                    'dateOfBirth',
                    'sex',
                    'id'
                ],
                include: [
                    {
                        model: models.PatientAdditionalData,
                        as: 'additionalData',
                        include: [
                            'ethnicity',
                            'medicalArea',
                            'nursingZone'
                        ]
                    },
                    'village'
                ]
            },
            {
                model: models.EncounterDiagnosis,
                as: 'diagnoses',
                include: [
                    'Diagnosis'
                ],
                attributes: [
                    'certainty',
                    'isPrimary'
                ],
                where: {
                    certainty: {
                        [_sequelize.Op.notIn]: [
                            _constants.DIAGNOSIS_CERTAINTY.DISPROVEN,
                            _constants.DIAGNOSIS_CERTAINTY.ERROR
                        ]
                    }
                },
                required: false
            },
            {
                model: models.Location,
                as: 'location',
                include: [
                    'locationGroup'
                ]
            },
            'examiner',
            'department'
        ],
        where: parametersToEncounterSqlWhere(parameters),
        order: [
            [
                'startDate',
                'ASC'
            ]
        ]
    });
    return encounters.map(convertModelToPlainObject);
};
const convertModelToPlainObject = (model)=>model.get({
        plain: true
    });
const getAllDiagnoses = async (models, encounters)=>{
    const newEncounters = [];
    for (const encounter of encounters){
        newEncounters.push({
            ...encounter,
            diagnoses: await models.EncounterDiagnosis.findAll({
                include: [
                    'Diagnosis'
                ],
                attributes: [
                    'certainty',
                    'isPrimary'
                ],
                where: {
                    certainty: {
                        [_sequelize.Op.notIn]: [
                            _constants.DIAGNOSIS_CERTAINTY.DISPROVEN,
                            _constants.DIAGNOSIS_CERTAINTY.ERROR
                        ]
                    },
                    encounterId: encounter.id
                },
                raw: true,
                nest: true
            })
        });
    }
    return newEncounters;
};
const stringifyDiagnoses = (diagnoses = [])=>diagnoses.map(({ Diagnosis, certainty })=>`${Diagnosis.name}: ${certainty}`).join(', ');
const transformDataPoint = (encounter)=>{
    const { patient, examiner, diagnoses } = encounter;
    const patientAdditionalData = patient.additionalData?.[0];
    const primaryDiagnoses = diagnoses.filter(({ isPrimary })=>isPrimary);
    const otherDiagnoses = diagnoses.filter(({ isPrimary })=>!isPrimary);
    return {
        firstName: patient.firstName,
        lastName: patient.lastName,
        displayId: patient.displayId,
        age: (0, _dateTime.ageInYears)(patient.dateOfBirth),
        sex: patient.sex,
        ethnicity: patientAdditionalData?.ethnicity?.name,
        contactPhone: patientAdditionalData?.primaryContactNumber,
        subdivision: patient.village?.name,
        medicalArea: patientAdditionalData?.medicalArea?.name,
        nursingZone: patientAdditionalData?.nursingZone?.name,
        clinician: examiner?.displayName,
        dateOfAttendance: (0, _dateTime.format)(encounter.startDate, 'dd-MM-yyyy'),
        department: encounter.department?.name,
        locationGroup: encounter.location?.locationGroup?.name,
        location: encounter.location?.name,
        reasonForAttendance: encounter.reasonForEncounter,
        primaryDiagnosis: stringifyDiagnoses(primaryDiagnoses),
        otherDiagnoses: stringifyDiagnoses(otherDiagnoses)
    };
};
const dataGenerator = async ({ models }, parameters = {})=>{
    let encounters = await getEncounters(models, parameters);
    if (parameters.diagnosis) {
        encounters = await getAllDiagnoses(models, encounters);
    }
    const reportData = encounters.map(transformDataPoint);
    return (0, _utilities.generateReportFromQueryData)(reportData, reportColumnTemplate);
};

//# sourceMappingURL=fiji-recent-attendance-list.js.map