"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "dataGenerator", {
    enumerable: true,
    get: function() {
        return dataGenerator;
    }
});
const _datefns = require("date-fns");
const _constants = require("@tamanu/constants");
const _dateTime = require("@tamanu/utils/dateTime");
const _utilities = require("./utilities");
const FIELDS = [
    'Patient ID',
    'Patient first name',
    'Patient last name',
    'DOB',
    'Age',
    'Sex',
    'Village',
    'Facility',
    'Department',
    'Area',
    'Request ID',
    'Request date and time',
    'Supervising clinician',
    'Requesting clinician',
    'Priority',
    'Imaging type',
    'Area to be imaged',
    'Status',
    'Reason for cancellation'
];
// Reason for cancellation is a configurable field but these are the default values
const DEFAULT_REASONS_FOR_CANCELLATION = {
    clinical: 'Clinical reason',
    duplicate: 'Duplicate',
    'entered-in-error': 'Entered in error',
    'patient-discharged': 'Patient discharged',
    'patient-refused': 'Patient refused',
    other: 'Other'
};
const reportColumnTemplate = FIELDS.map((field)=>{
    if (field === 'Status') {
        return {
            title: field,
            accessor: (data)=>_constants.IMAGING_REQUEST_STATUS_CONFIG[data[field]] ? _constants.IMAGING_REQUEST_STATUS_CONFIG[data[field]].label : data[field]
        };
    }
    if (field === 'Reason for cancellation') {
        return {
            title: field,
            accessor: (data)=>DEFAULT_REASONS_FOR_CANCELLATION[data[field]] ?? data[field]
        };
    }
    return {
        title: field,
        accessor: (data)=>data[field]
    };
});
const query = `
select
  p.display_id as "Patient ID",
  p.first_name as "Patient first name" ,
  p.last_name as "Patient last name",
  to_char(p.date_of_birth ::timestamp::date, 'DD/MM/YYYY') as "DOB",
  date_part('year', age(p.date_of_birth::date)) as "Age",
  p.sex as "Sex",
  rdv.name as "Village",
  f.name as "Facility",
  d.name as "Department",
  locationGroup.name as "Area",
  ir.display_id as "Request ID",
  to_char(ir.requested_date::timestamp, 'DD/MM/YYYY HH12:MI AM') as "Request date and time",
  u_supervising.display_name as "Supervising clinician",
  u_requesting.display_name as "Requesting clinician",
  ir.priority as "Priority",
  ir.imaging_type as "Imaging type",
  case
    when ira.id is not null then rdi.name
    else n.content
    end as "Area to be imaged",
  ir.status as "Status",
  case
      when ir.status = 'cancelled' then ir.reason_for_cancellation
      else null
      end as "Reason for cancellation"
from
  imaging_requests ir
  left join encounters e on e.id=ir.encounter_id
  left join patients p on p.id=e.patient_id
  left join reference_data rdv on rdv.id=p.village_id
  left join locations l on l.id=e.location_id
  left join location_groups locationGroup on l.location_group_id=locationGroup.id
  left join facilities f on f.id = l.facility_id
  left join departments d on d.id = e.department_id
  left join users u_supervising on u_supervising.id=e.examiner_id
  left join users u_requesting on u_requesting.id=ir.requested_by_id
  left join notes n on n.record_id = ir.id and n.note_type = 'areaToBeImaged'
  left join imaging_request_areas ira on ira.imaging_request_id = ir.id
  left join reference_data rdi on rdi.id = ira.area_id
where
  case when :from_date is not null then ir.requested_date::date >= :from_date::date else true end
  and case when :to_date is not null then ir.requested_date::date <= :to_date::date else true end
  and case when :requested_by_id is not null then ir.requested_by_id = :requested_by_id else true end
  and case when :imaging_type is not null then ir.imaging_type = :imaging_type else true end
  and case when :areStatuses is not null then ir.status IN(:statuses) else true end
order by ir.requested_date;
`;
const getData = async (sequelize, parameters)=>{
    const { fromDate = (0, _dateTime.toDateTimeString)((0, _datefns.subDays)(new Date(), 30)), toDate, requestedById, imagingType, statuses } = parameters;
    return sequelize.query(query, {
        type: sequelize.QueryTypes.SELECT,
        replacements: {
            from_date: fromDate ?? null,
            to_date: toDate ?? null,
            requested_by_id: requestedById ?? null,
            imaging_type: imagingType ?? null,
            statuses: statuses ?? null,
            areStatuses: statuses ? 'true' : null
        }
    });
};
const dataGenerator = async ({ sequelize }, parameters = {})=>{
    const results = await getData(sequelize, parameters);
    return (0, _utilities.generateReportFromQueryData)(results, reportColumnTemplate);
};

//# sourceMappingURL=imaging-requests-line-list.js.map