"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "GenerateMedicationAdministrationRecords", {
    enumerable: true,
    get: function() {
        return GenerateMedicationAdministrationRecords;
    }
});
const _config = /*#__PURE__*/ _interop_require_default(require("config"));
const _tasks = require("@tamanu/shared/tasks");
const _logging = require("@tamanu/shared/services/logging");
const _sequelize = require("sequelize");
const _sleepAsync = require("@tamanu/utils/sleepAsync");
const _ = require(".");
const _dateTime = require("@tamanu/utils/dateTime");
function _interop_require_default(obj) {
    return obj && obj.__esModule ? obj : {
        default: obj
    };
}
let GenerateMedicationAdministrationRecords = class GenerateMedicationAdministrationRecords extends _tasks.ScheduledTask {
    getName() {
        return 'GenerateMedicationAdministrationRecords';
    }
    async run() {
        await this.cleanupInvalidMedicationAdministrationRecords();
        await this.generateMedicationAdministrationRecords();
    }
    async cleanupInvalidMedicationAdministrationRecords() {
        const { MedicationAdministrationRecord } = this.models;
        await MedicationAdministrationRecord.removeInvalidMedicationAdministrationRecords();
    }
    async generateMedicationAdministrationRecords() {
        const { Prescription, MedicationAdministrationRecord } = this.models;
        const baseQueryOptions = {
            where: {
                endDate: {
                    [_sequelize.Op.or]: [
                        {
                            [_sequelize.Op.gt]: (0, _dateTime.getCurrentDateTimeString)()
                        },
                        {
                            [_sequelize.Op.is]: null
                        }
                    ]
                },
                discontinued: {
                    [_sequelize.Op.not]: true
                }
            },
            include: [
                {
                    model: this.models.EncounterPrescription,
                    required: true,
                    as: 'encounterPrescription',
                    include: [
                        {
                            model: this.models.Encounter,
                            as: 'encounter',
                            // do not generate MARs for encounters that are discharged
                            where: {
                                endDate: null
                            }
                        }
                    ]
                }
            ]
        };
        const toProcess = await Prescription.count({
            ...baseQueryOptions
        });
        if (toProcess === 0) return;
        const { batchSize, batchSleepAsyncDurationInMilliseconds } = this.config;
        if (!batchSize || !batchSleepAsyncDurationInMilliseconds) {
            throw new _.InvalidConfigError('batchSize and batchSleepAsyncDurationInMilliseconds must be set for GenerateMedicationAdministrationRecords');
        }
        const batchCount = Math.ceil(toProcess / batchSize);
        _logging.log.info('Running batched generating medication administration records', {
            recordCount: toProcess,
            batchCount,
            batchSize
        });
        for(let i = 0; i < batchCount; i++){
            const prescriptions = await Prescription.findAll({
                ...baseQueryOptions,
                limit: batchSize
            });
            for (const prescription of prescriptions){
                try {
                    await MedicationAdministrationRecord.generateMedicationAdministrationRecords(prescription);
                } catch (error) {
                    _logging.log.error('Failed to generate medication administration records', {
                        prescriptionId: prescription.id,
                        error: error.message
                    });
                }
            }
            await (0, _sleepAsync.sleepAsync)(batchSleepAsyncDurationInMilliseconds);
        }
    }
    /**
   *
   * @param {import('../ApplicationContext').ApplicationContext} context
   */ constructor(context){
        const conf = _config.default.schedules.generateMedicationAdministrationRecords;
        const { schedule, jitterTime, enabled } = conf;
        super(schedule, _logging.log, jitterTime, enabled);
        this.models = context.store.models;
        this.config = conf;
        this.sequelize = context.store.sequelize;
    }
};

//# sourceMappingURL=GenerateMedicationAdministrationRecords.js.map