"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
function _export(target, all) {
    for(var name in all)Object.defineProperty(target, name, {
        enumerable: true,
        get: all[name]
    });
}
_export(exports, {
    PERMISSIONS: function() {
        return PERMISSIONS;
    },
    programImporter: function() {
        return programImporter;
    }
});
const _logging = require("@tamanu/shared/services/logging");
const _xlsx = require("xlsx");
const _importRows = require("../importer/importRows");
const _readMetadata = require("./readMetadata");
const _importSurvey = require("./importSurvey");
const _importProgramRegistry = require("./importProgramRegistry");
const PERMISSIONS = [
    'Program',
    'Survey'
];
async function programImporter({ errors, models, stats, file, whitelist = [], data, checkPermission }) {
    checkPermission('create', 'Program');
    checkPermission('write', 'Program');
    const createContext = (sheetName)=>({
            errors,
            log: _logging.log.child({
                file,
                sheetName
            }),
            models
        });
    _logging.log.info('Importing surveys from file', {
        file
    });
    const workbook = data ? (0, _xlsx.read)(data, {
        type: 'buffer'
    }) : (0, _xlsx.readFile)(file);
    const { programRecord, surveyMetadata } = (0, _readMetadata.readMetadata)(workbook.Sheets.Metadata);
    // actually import the program to the database
    stats.push(await (0, _importRows.importRows)(createContext('Metadata'), {
        sheetName: 'Metadata',
        rows: [
            {
                model: 'Program',
                values: programRecord,
                sheetRow: 0
            }
        ]
    }));
    // Note - the program registry must be imported before the surveys
    // in order to properly validate them.
    stats.push(await (0, _importProgramRegistry.importProgramRegistry)(createContext('ProgramRegistry'), workbook, programRecord.id));
    const surveysToImport = surveyMetadata.filter(({ name, code })=>{
        // check against whitelist
        if (!whitelist || whitelist.length === 0) {
            return true;
        }
        return whitelist.some((x)=>x === name || x === code);
    });
    _logging.log.debug('Importing surveys', {
        count: surveysToImport.length
    });
    if (surveysToImport.length) {
        checkPermission('create', 'Survey');
        checkPermission('write', 'Survey');
    }
    // then loop over each survey defined in metadata and import it
    for (const surveyInfo of surveysToImport){
        try {
            const context = createContext(surveyInfo.name);
            const result = await (0, _importSurvey.importSurvey)(context, workbook, surveyInfo);
            stats.push(result);
        } catch (e) {
            errors.push(e);
        }
    }
    _logging.log.debug('Done importing programs data');
}

//# sourceMappingURL=programImporter.js.map