"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "Problem", {
    enumerable: true,
    get: function() {
        return Problem;
    }
});
const _lodash = require("lodash");
const _yup = require("yup");
const _core = require("zod/v4/core");
const _BaseError = require("./BaseError");
const _errors = require("./errors");
const _constants = require("./constants");
const _splitUpStack = require("./splitUpStack");
const LINK = '/problems/';
const IANA = 'https://iana.org/assignments/http-problem-types#';
let Problem = class Problem extends Error {
    type;
    title;
    status;
    detail;
    extra = new Map();
    /** When the Problem has been obtained from an API call, this will typically have its Response object. */ response;
    constructor(type, title, status = 500, detail){
        super(detail ?? title);
        this.type = type;
        this.title = title;
        this.status = status;
        this.detail = detail;
    }
    static fromError(error) {
        if (error instanceof _yup.ValidationError || error instanceof _core.$ZodError) {
            error = new _errors.ValidationError(error.message).withCause(error);
        }
        if (error instanceof _BaseError.BaseError) {
            const problem = new Problem(error.type, error.title, error.status, error.detail);
            for (const [key, value] of Object.entries(error.extraData)){
                problem.extra.set((0, _lodash.kebabCase)(key), value);
            }
            if (error.stack) {
                problem.extra.set('stack', (0, _splitUpStack.splitUpStack)(error.stack));
            }
            return problem;
        }
        return new Problem(_constants.ERROR_TYPE.UNKNOWN, error.name, 500, error.message);
    }
    get headers() {
        const headers = {
            'content-type': 'application/problem+json'
        };
        if (this.extra.has('retry-after')) {
            headers['Retry-After'] = this.extra.get('retry-after').toString();
        }
        return headers;
    }
    /** Mostly for test mocks, converts a Problem into a fetch Response. */ intoResponse() {
        const body = JSON.stringify(this.toJSON());
        const headers = this.headers;
        return new Response(body, {
            status: this.status,
            headers
        });
    }
    excludeSensitiveFields(exclude) {
        if (exclude) {
            this.extra.delete('stack');
            this.extra.delete('request-url');
        }
        return this;
    }
    toJSON() {
        return {
            ...Object.fromEntries(this.extra.entries()),
            type: (0, _constants.isKnownErrorType)(this.type) ? _constants.IANA_TYPES.includes(this.type) ? `${IANA}${this.type}` : `${LINK}${this.type}` : this.type,
            title: this.title,
            status: this.status,
            detail: this.detail === this.title ? undefined : this.detail
        };
    }
    static fromJSON(json) {
        let type = json.type;
        if (!type) {
            return null;
        }
        if (json.type.startsWith(LINK)) {
            const slug = json.type.replace(new RegExp(`^${(0, _lodash.escapeRegExp)(LINK)}`), '');
            if (slug && (0, _constants.isKnownErrorType)(slug)) {
                type = slug;
            }
        } else if (json.type.startsWith(IANA)) {
            const slug = json.type.replace(new RegExp(`^${(0, _lodash.escapeRegExp)(IANA)}`), '');
            if (slug && (0, _constants.isKnownErrorType)(slug)) {
                type = slug;
            }
        }
        const problem = new Problem(type, json.title, json.status, json.detail);
        for (const [key, value] of Object.entries(json)){
            if (_constants.WELL_KNOWN_PROBLEM_KEYS.includes(key)) continue;
            problem.extra.set((0, _lodash.kebabCase)(key), value);
        }
        return problem;
    }
};

//# sourceMappingURL=Problem.js.map