"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "medication", {
    enumerable: true,
    get: function() {
        return medication;
    }
});
const _express = /*#__PURE__*/ _interop_require_default(require("express"));
const _expressasynchandler = /*#__PURE__*/ _interop_require_default(require("express-async-handler"));
const _dateTime = require("@tamanu/utils/dateTime");
const _zod = require("zod");
const _errors = require("@tamanu/errors");
const _constants = require("@tamanu/constants");
const _datefns = require("date-fns");
const _sequelize = require("sequelize");
const _validate = require("../../utils/validate");
function _interop_require_default(obj) {
    return obj && obj.__esModule ? obj : {
        default: obj
    };
}
const medication = _express.default.Router();
const checkSensitiveMedicationPermission = async (medicationIds, req, action)=>{
    if (!medicationIds?.length) return true;
    const isSensitive = await req.models.ReferenceDrug.hasSensitiveMedication(medicationIds);
    if (isSensitive) {
        req.checkPermission(action, 'SensitiveMedication');
    }
};
medication.get('/:id', (0, _expressasynchandler.default)(async (req, res)=>{
    const { models, params, query } = req;
    const { Prescription } = models;
    req.checkPermission('read', 'Medication');
    const object = await Prescription.findByPk(params.id, {
        include: Prescription.getFullReferenceAssociations()
    });
    await checkSensitiveMedicationPermission([
        object.medicationId
    ], req, 'read');
    if (!object) throw new _errors.NotFoundError();
    if (object) {
        await req.audit.access({
            recordId: object.id,
            frontEndContext: params,
            model: Prescription,
            facilityId: query.facilityId
        });
    }
    res.send(object);
}));
const medicationInputSchema = _zod.z.object({
    encounterId: _zod.z.string().optional().nullable(),
    patientId: _zod.z.string().optional().nullable(),
    date: _dateTime.dateCustomValidation,
    notes: _zod.z.string().optional().nullable(),
    indication: _zod.z.string().optional().nullable(),
    route: _zod.z.enum(Object.values(_constants.DRUG_ROUTES)),
    medicationId: _zod.z.string(),
    prescriberId: _zod.z.string(),
    quantity: _zod.z.coerce.number().int().optional().nullable(),
    isOngoing: _zod.z.boolean().optional().nullable(),
    isPrn: _zod.z.boolean().optional().nullable(),
    isVariableDose: _zod.z.boolean().optional().nullable(),
    doseAmount: _zod.z.coerce.number().positive().optional().nullable(),
    units: _zod.z.enum(Object.values(_constants.DRUG_UNITS)),
    frequency: _zod.z.enum(Object.values(_constants.ADMINISTRATION_FREQUENCIES)),
    startDate: _dateTime.datetimeCustomValidation,
    durationValue: _zod.z.coerce.number().positive().optional().nullable(),
    durationUnit: _zod.z.enum(Object.values(_constants.MEDICATION_DURATION_UNITS)).optional().nullable(),
    isPhoneOrder: _zod.z.boolean().optional(),
    idealTimes: _zod.z.array(_zod.z.string()).optional().nullable()
}).strip().superRefine((val, ctx)=>{
    if (!val.isVariableDose && !val.doseAmount) {
        ctx.addIssue({
            code: _zod.z.ZodIssueCode.custom,
            message: 'Dose amount is required or isVariableDose must be true'
        });
    }
    if (val.durationValue && !val.durationUnit) {
        ctx.addIssue({
            code: _zod.z.ZodIssueCode.custom,
            message: 'Duration unit is required when duration value is provided'
        });
    }
    if (val.durationUnit && !val.durationValue) {
        ctx.addIssue({
            code: _zod.z.ZodIssueCode.custom,
            message: 'Duration value is required when duration unit is provided'
        });
    }
    if (val.frequency !== _constants.ADMINISTRATION_FREQUENCIES.IMMEDIATELY && val.frequency !== _constants.ADMINISTRATION_FREQUENCIES.AS_DIRECTED && !val.idealTimes?.length) {
        ctx.addIssue({
            code: _zod.z.ZodIssueCode.custom,
            message: 'Ideal times are required when frequency is not IMMEDIATELY or AS_DIRECTED'
        });
    }
    if ((val.frequency === _constants.ADMINISTRATION_FREQUENCIES.IMMEDIATELY || val.isOngoing) && val.durationValue) {
        ctx.addIssue({
            code: _zod.z.ZodIssueCode.custom,
            message: 'Duration value and unit are not allowed when frequency is IMMEDIATELY or isOngoing'
        });
    }
});
medication.post('/patientOngoingPrescription/:patientId', (0, _expressasynchandler.default)(async (req, res)=>{
    const { models, db } = req;
    const patientId = req.params.patientId;
    const { Prescription, Patient, PatientOngoingPrescription } = models;
    req.checkPermission('create', 'Medication');
    const data = await medicationInputSchema.parseAsync(req.body);
    await checkSensitiveMedicationPermission([
        data.medicationId
    ], req, 'create');
    const patient = await Patient.findByPk(patientId);
    if (!patient) {
        throw new _errors.InvalidOperationError(`Patient with id ${patientId} not found`);
    }
    const result = await db.transaction(async (transaction)=>{
        const prescription = await Prescription.create(data, {
            transaction
        });
        await PatientOngoingPrescription.create({
            patientId,
            prescriptionId: prescription.id
        }, {
            transaction
        });
        return prescription;
    });
    res.send(result.forResponse());
}));
const createEncounterPrescription = async ({ encounter, data, models })=>{
    const { Prescription, EncounterPrescription, MedicationAdministrationRecord } = models;
    const prescription = await Prescription.create({
        ...data,
        id: undefined
    });
    await EncounterPrescription.create({
        encounterId: encounter.id,
        prescriptionId: prescription.id
    });
    await MedicationAdministrationRecord.generateMedicationAdministrationRecords(prescription);
    return prescription;
};
medication.post('/encounterPrescription/:encounterId', (0, _expressasynchandler.default)(async (req, res)=>{
    const { models, db } = req;
    const encounterId = req.params.encounterId;
    const { Encounter } = models;
    req.checkPermission('create', 'Medication');
    const data = await medicationInputSchema.parseAsync(req.body);
    await checkSensitiveMedicationPermission([
        data.medicationId
    ], req, 'create');
    const encounter = await Encounter.findByPk(encounterId);
    if (!encounter) {
        throw new _errors.InvalidOperationError(`Encounter with id ${encounterId} not found`);
    }
    if (encounter.endDate) {
        throw new _errors.InvalidOperationError(`Encounter with id ${encounterId} is discharged`);
    }
    if (new Date(data.startDate) < new Date(encounter.startDate)) {
        throw new _errors.InvalidOperationError(`Cannot create prescription with start date (${data.startDate}) before encounter start date (${encounter.startDate})`);
    }
    const result = await db.transaction(async ()=>{
        const prescription = await createEncounterPrescription({
            encounter,
            data,
            models
        });
        return prescription;
    });
    res.send(result.forResponse());
}));
medication.post('/medication-set', (0, _expressasynchandler.default)(async (req, res)=>{
    req.checkPermission('create', 'Medication');
    const { models } = req;
    const { medicationSet, encounterId } = req.body;
    const { Encounter } = models;
    const encounter = await Encounter.findByPk(encounterId);
    if (!encounter) {
        throw new _errors.InvalidOperationError(`Encounter with id ${encounterId} not found`);
    }
    if (encounter.endDate) {
        throw new _errors.InvalidOperationError(`Encounter with id ${encounterId} is discharged`);
    }
    for (const medication of medicationSet){
        if (new Date(medication.startDate) < new Date(encounter.startDate)) {
            throw new _errors.InvalidOperationError(`Cannot create prescription with start date (${medication.startDate}) before encounter start date (${encounter.startDate})`);
        }
    }
    await checkSensitiveMedicationPermission(medicationSet.map((m)=>m.medicationId), req, 'create');
    const result = await req.db.transaction(async ()=>{
        const prescriptions = [];
        for (const medication of medicationSet){
            const data = await medicationInputSchema.parseAsync(medication);
            const prescription = await createEncounterPrescription({
                encounter,
                data,
                models
            });
            prescriptions.push(prescription);
        }
        return prescriptions;
    });
    res.send(result.map((prescription)=>prescription.forResponse()));
}));
const importOngoingMedicationsSchema = _zod.z.object({
    encounterId: _zod.z.uuid({
        message: 'Valid encounter ID is required'
    }),
    prescriptionIds: _zod.z.array(_zod.z.uuid({
        message: 'Valid prescription ID is required'
    })),
    prescriberId: _zod.z.string()
}).strip();
medication.post('/import-ongoing', (0, _expressasynchandler.default)(async (req, res)=>{
    const { models, db } = req;
    const { Encounter, Prescription, PatientOngoingPrescription } = models;
    const { prescriptionIds, prescriberId, encounterId } = (0, _validate.validate)(importOngoingMedicationsSchema, req.body);
    req.checkPermission('create', 'Medication');
    const encounter = await Encounter.findByPk(encounterId);
    if (!encounter) {
        throw new _errors.InvalidOperationError(`Encounter with id ${encounterId} not found`);
    }
    if (encounter.endDate) {
        throw new _errors.InvalidOperationError(`Encounter with id ${encounterId} is discharged`);
    }
    const ongoingPrescriptions = await Prescription.findAll({
        where: {
            id: {
                [_sequelize.Op.in]: prescriptionIds
            },
            discontinued: {
                [_sequelize.Op.not]: true
            }
        },
        include: [
            {
                model: PatientOngoingPrescription,
                as: 'patientOngoingPrescription',
                where: {
                    patientId: encounter.patientId
                }
            }
        ]
    });
    if (ongoingPrescriptions.length !== prescriptionIds.length) {
        const foundIds = new Set(ongoingPrescriptions.map((p)=>p.id));
        const missingIds = prescriptionIds.filter((id)=>!foundIds.has(id));
        throw new _errors.InvalidOperationError(`Prescription(s) with id(s) ${missingIds.join(', ')} not found, have been discontinued, or do not belong to this patient.`);
    }
    const importPrescriptions = ongoingPrescriptions;
    await checkSensitiveMedicationPermission(importPrescriptions.map((prescription)=>prescription.medicationId), req, 'create');
    const result = await db.transaction(async ()=>{
        const newPrescriptions = [];
        for (const prescription of importPrescriptions){
            const newPrescription = await createEncounterPrescription({
                encounter,
                data: {
                    ...prescription.toJSON(),
                    createdAt: undefined,
                    updatedAt: undefined,
                    prescriberId,
                    date: (0, _dateTime.getCurrentDateTimeString)(),
                    startDate: (0, _dateTime.getCurrentDateTimeString)()
                },
                models
            });
            newPrescriptions.push(newPrescription);
        }
        return newPrescriptions;
    });
    res.send({
        count: result.length,
        data: result.map((prescription)=>prescription.forResponse())
    });
}));
const updatePharmacyNotesInputSchema = _zod.z.object({
    pharmacyNotes: _zod.z.string().optional().nullable().transform((v)=>!v ? null : v),
    displayPharmacyNotesInMar: _zod.z.boolean().optional()
}).strip();
medication.put('/:id/pharmacy-notes', (0, _expressasynchandler.default)(async (req, res)=>{
    const { models, params } = req;
    const { Prescription } = models;
    const { pharmacyNotes, displayPharmacyNotesInMar } = await updatePharmacyNotesInputSchema.parseAsync(req.body);
    req.checkPermission('write', 'Medication');
    req.checkPermission('create', 'MedicationPharmacyNote');
    const prescription = await Prescription.findByPk(params.id);
    if (!prescription) {
        throw new _errors.InvalidOperationError(`Prescription with id ${params.id} not found`);
    }
    if (prescription.pharmacyNotes && prescription.pharmacyNotes !== pharmacyNotes) {
        req.checkPermission('write', 'MedicationPharmacyNote');
    }
    await checkSensitiveMedicationPermission([
        prescription.medicationId
    ], req, 'write');
    prescription.pharmacyNotes = pharmacyNotes;
    prescription.displayPharmacyNotesInMar = displayPharmacyNotesInMar;
    await prescription.save();
    res.send(prescription.forResponse());
}));
const discontinueInputSchema = _zod.z.object({
    discontinuingClinicianId: _zod.z.string(),
    discontinuingReason: _zod.z.string().optional(),
    discontinuingDate: _dateTime.datetimeCustomValidation.optional()
}).strip();
medication.post('/:id/discontinue', (0, _expressasynchandler.default)(async (req, res)=>{
    const { models, params, db } = req;
    const { Encounter, Prescription } = models;
    const data = await discontinueInputSchema.parseAsync(req.body);
    req.checkPermission('write', 'Medication');
    const prescription = await Prescription.findByPk(params.id, {
        include: [
            {
                model: models.EncounterPrescription,
                as: 'encounterPrescription',
                attributes: [
                    'encounterId'
                ],
                required: false
            }
        ]
    });
    if (!prescription) {
        throw new _errors.InvalidOperationError(`Prescription with id ${params.id} not found`);
    }
    await checkSensitiveMedicationPermission([
        prescription.medicationId
    ], req, 'write');
    if (prescription.discontinued) {
        throw new _errors.EditConflictError('Prescription already discontinued');
    }
    await db.transaction(async ()=>{
        Object.assign(prescription, {
            ...data,
            discontinuedDate: data.discontinuingDate ? (0, _dateTime.toDateTimeString)(data.discontinuingDate) : (0, _dateTime.getCurrentDateTimeString)(),
            discontinued: true
        });
        await prescription.save();
        // if the prescription is associated with an encounter, we need to remove the same prescription from the patient's ongoing medications
        const encounterId = prescription.encounterPrescription?.encounterId;
        if (!encounterId) return;
        const encounter = await Encounter.findByPk(encounterId);
        // if the encounter is not found or the encounter is ended, we don't need to remove the prescription from the patient's ongoing medications
        if (!encounter || encounter.endDate) return;
        const existingPatientOngoingPrescription = await models.PatientOngoingPrescription.findPatientOngoingPrescriptionWithSameDetails(encounter.patientId, prescription);
        if (existingPatientOngoingPrescription) {
            const existingOngoingPrescription = existingPatientOngoingPrescription.prescription;
            Object.assign(existingOngoingPrescription, {
                discontinuingClinicianId: _constants.SYSTEM_USER_UUID,
                discontinuedDate: (0, _dateTime.getCurrentDateTimeString)(),
                discontinued: true,
                discontinuingReason: 'Discontinued by user discontinue encounter prescription'
            });
            await existingOngoingPrescription.save();
        }
    });
    const updatedObject = await Prescription.findByPk(params.id, {
        include: Prescription.getListReferenceAssociations()
    });
    res.send(updatedObject.forResponse());
}));
const pauseMedicationSchema = _zod.z.object({
    encounterId: _zod.z.string().uuid({
        message: 'Valid encounter ID is required'
    }),
    pauseDuration: _zod.z.coerce.number().positive({
        message: 'Pause duration must be a positive number'
    }),
    pauseTimeUnit: _zod.z.enum(Object.keys(_constants.MEDICATION_PAUSE_DURATION_UNITS_LABELS), {
        errorMap: ()=>({
                message: 'Pause time unit must be either "Hours" or "Days"'
            })
    }),
    notes: _zod.z.string().optional(),
    pauseStartDate: _dateTime.datetimeCustomValidation.optional()
}).strip();
// Pause a medication
medication.post('/:id/pause', (0, _expressasynchandler.default)(async (req, res)=>{
    const { models, params, user } = req;
    const { Prescription, EncounterPrescription, EncounterPausePrescription } = models;
    // Validate request body against the schema
    const { encounterId, pauseDuration, pauseTimeUnit, notes, pauseStartDate: pauseStartDateInput } = await pauseMedicationSchema.parseAsync(req.body);
    req.checkPermission('write', 'Medication');
    // Find the prescription
    const prescription = await Prescription.findByPk(params.id);
    if (!prescription) {
        throw new _errors.InvalidOperationError(`Prescription with id ${params.id} not found`);
    }
    await checkSensitiveMedicationPermission([
        prescription.medicationId
    ], req, 'write');
    if (prescription.frequency === _constants.ADMINISTRATION_FREQUENCIES.IMMEDIATELY) {
        throw new _errors.InvalidOperationError(`Medication with frequency ${_constants.ADMINISTRATION_FREQUENCIES.IMMEDIATELY} cannot be paused`);
    }
    // Find the encounter prescription link
    const encounterPrescription = await EncounterPrescription.findOne({
        where: {
            prescriptionId: params.id,
            encounterId
        }
    });
    if (!encounterPrescription) {
        throw new _errors.InvalidOperationError(`Prescription is not associated with the provided encounter`);
    }
    // Check if the medication is already paused using the static method
    const { isPaused } = await EncounterPausePrescription.isPrescriptionPaused(params.id);
    if (isPaused) {
        throw new _errors.InvalidOperationError(`Medication is already paused`);
    }
    // Calculate the pause end date to validate against prescription end date
    const pauseStartDate = pauseStartDateInput ? (0, _dateTime.toDateTimeString)(pauseStartDateInput) : (0, _dateTime.getCurrentDateTimeString)();
    const pauseEndDate = (0, _datefns.add)(new Date(pauseStartDate), {
        [pauseTimeUnit]: pauseDuration
    });
    // Validate that pause duration doesn't extend beyond medication end date
    if (prescription.endDate && (0, _datefns.isAfter)(pauseEndDate, new Date(prescription.endDate))) {
        throw new _errors.InvalidOperationError('Pause duration extends beyond the medication end date. Please reduce the pause duration or choose a different medication.');
    }
    // Create pause record (pauseEndDate will be calculated by model hooks)
    const pauseRecord = await EncounterPausePrescription.create({
        encounterPrescriptionId: encounterPrescription.id,
        pauseDuration,
        pauseTimeUnit,
        pauseStartDate,
        pauseEndDate: (0, _dateTime.toDateTimeString)(pauseEndDate),
        notes,
        pausingClinicianId: user.id,
        createdBy: user.id
    });
    // Return the pause record along with the prescription
    res.send({
        prescription: prescription.forResponse(),
        pauseRecord: pauseRecord.forResponse()
    });
}));
const resumeMedicationSchema = _zod.z.object({
    encounterId: _zod.z.string().uuid({
        message: 'Valid encounter ID is required'
    })
}).strip();
// Resume a paused medication
medication.post('/:id/resume', (0, _expressasynchandler.default)(async (req, res)=>{
    const { models, params, user } = req;
    const { Prescription, EncounterPrescription, EncounterPausePrescription } = models;
    // Validate request body against the schema
    const { encounterId } = await resumeMedicationSchema.parseAsync(req.body);
    req.checkPermission('write', 'Medication');
    // Find the prescription
    const prescription = await Prescription.findByPk(params.id);
    if (!prescription) {
        throw new _errors.InvalidOperationError(`Prescription with id ${params.id} not found`);
    }
    await checkSensitiveMedicationPermission([
        prescription.medicationId
    ], req, 'write');
    // Find the encounter prescription link
    const encounterPrescription = await EncounterPrescription.findOne({
        where: {
            prescriptionId: params.id,
            encounterId
        }
    });
    if (!encounterPrescription) {
        throw new _errors.InvalidOperationError(`Prescription is not associated with the provided encounter`);
    }
    // Check if the medication is currently paused
    const { isPaused, pauseData } = await EncounterPausePrescription.isPrescriptionPaused(params.id);
    if (!isPaused || !pauseData) {
        throw new _errors.InvalidOperationError(`Medication is not currently paused`);
    }
    // Update pause record to end now (history is created automatically by model hooks)
    const currentDate = (0, _dateTime.getCurrentDateTimeString)();
    pauseData.pauseEndDate = currentDate;
    pauseData.updatedBy = user.id;
    await pauseData.save();
    // Return the updated prescription
    res.send({
        prescription: prescription.forResponse(),
        pauseRecord: pauseData.forResponse()
    });
}));
const getPauseQuerySchema = _zod.z.object({
    encounterId: _zod.z.string().uuid({
        message: 'Valid encounter ID is required'
    })
}).strip();
// Get active pause information for a medication
medication.get('/:id/pause', (0, _expressasynchandler.default)(async (req, res)=>{
    const { models, params } = req;
    const { Prescription, EncounterPrescription, EncounterPausePrescription } = models;
    // Validate query params against the schema
    const { encounterId } = await getPauseQuerySchema.parseAsync(req.query);
    req.checkPermission('read', 'Medication');
    // Find the prescription
    const prescription = await Prescription.findByPk(params.id);
    if (!prescription) {
        throw new _errors.InvalidOperationError(`Prescription with id ${params.id} not found`);
    }
    // Find the encounter prescription link - needed for returning the correct pause
    const encounterPrescription = await EncounterPrescription.findOne({
        where: {
            prescriptionId: params.id,
            encounterId
        }
    });
    if (!encounterPrescription) {
        throw new _errors.InvalidOperationError(`Prescription is not associated with the provided encounter`);
    }
    // Check if the medication is paused using our static method
    const { isPaused, pauseData } = await EncounterPausePrescription.isPrescriptionPaused(params.id);
    if (!isPaused || !pauseData) {
        // Not paused - return null
        return res.send({
            pauseRecord: null
        });
    }
    // Load associations for the pause record
    await pauseData.reload({
        include: [
            {
                association: 'pausingClinician',
                attributes: [
                    'id',
                    'displayName'
                ]
            }
        ]
    });
    // Return active pause record
    res.send({
        pauseRecord: pauseData.forResponse()
    });
}));
const getPausesQuerySchema = _zod.z.object({
    encounterId: _zod.z.string().uuid({
        message: 'Valid encounter ID is required'
    }),
    marDate: _zod.z.string().optional().refine((val)=>!val || !isNaN(new Date(val).getTime()), {
        message: 'marDate must be a valid date string'
    })
}).strip();
medication.get('/:id/pauses', (0, _expressasynchandler.default)(async (req, res)=>{
    const { models, params } = req;
    const { Prescription, EncounterPrescription, EncounterPausePrescription } = models;
    // Validate query params against the schema
    const { encounterId, marDate } = await getPausesQuerySchema.parseAsync(req.query);
    req.checkPermission('read', 'Medication');
    // Find the prescription
    const prescription = await Prescription.findByPk(params.id);
    if (!prescription) {
        throw new _errors.InvalidOperationError(`Prescription with id ${params.id} not found`);
    }
    // Find the encounter prescription link
    const encounterPrescription = await EncounterPrescription.findOne({
        where: {
            prescriptionId: params.id,
            encounterId
        }
    });
    if (!encounterPrescription) {
        throw new _errors.InvalidOperationError(`Prescription is not associated with the provided encounter`);
    }
    // Build where clause
    const whereClause = {
        encounterPrescriptionId: encounterPrescription.id
    };
    // If marDate is provided, filter for pauses that are active on that date
    if (marDate) {
        const startOfMarDate = (0, _datefns.format)(new Date(marDate), 'yyyy-MM-dd 00:00:00');
        const endOfMarDate = (0, _datefns.format)(new Date(marDate), 'yyyy-MM-dd 23:59:59');
        whereClause[_sequelize.Op.and] = [
            {
                pauseStartDate: {
                    [_sequelize.Op.lte]: endOfMarDate
                }
            },
            {
                pauseEndDate: {
                    [_sequelize.Op.gte]: startOfMarDate
                }
            }
        ];
    }
    // Get all pause records for this encounter prescription with filters
    const pauseRecords = await EncounterPausePrescription.findAll({
        where: whereClause,
        order: [
            [
                'pauseEndDate',
                'DESC'
            ]
        ]
    });
    // Return pause records
    res.send({
        count: pauseRecords.length,
        data: pauseRecords.map((record)=>record.forResponse())
    });
}));
const pauseHistoryQuerySchema = _zod.z.object({
    encounterId: _zod.z.string().uuid({
        message: 'Valid encounter ID is required'
    })
}).strip();
// Get pause history for a medication
medication.get('/:id/pause-history', (0, _expressasynchandler.default)(async (req, res)=>{
    const { models, params } = req;
    const { Prescription, EncounterPrescription, EncounterPausePrescriptionHistory } = models;
    // Validate query params against the schema
    const { encounterId } = await pauseHistoryQuerySchema.parseAsync(req.query);
    req.checkPermission('read', 'Medication');
    // Find the prescription
    const prescription = await Prescription.findByPk(params.id);
    if (!prescription) {
        throw new _errors.InvalidOperationError(`Prescription with id ${params.id} not found`);
    }
    // Find the encounter prescription link
    const encounterPrescription = await EncounterPrescription.findOne({
        where: {
            prescriptionId: params.id,
            encounterId
        }
    });
    if (!encounterPrescription) {
        throw new _errors.InvalidOperationError(`Prescription is not associated with the provided encounter`);
    }
    // Get pause history for this encounter prescription
    const pauseHistory = await EncounterPausePrescriptionHistory.findAll({
        where: {
            encounterPrescriptionId: encounterPrescription.id
        },
        order: [
            [
                'actionDate',
                'DESC'
            ]
        ]
    });
    // Return history records
    res.send({
        count: pauseHistory.length,
        data: pauseHistory.map((record)=>record.forResponse())
    });
}));
const givenMarUpdateSchema = _zod.z.object({
    dose: _zod.z.object({
        doseAmount: _zod.z.number(),
        givenTime: _dateTime.datetimeCustomValidation,
        givenByUserId: _zod.z.string().optional(),
        recordedByUserId: _zod.z.string().optional()
    }),
    recordedByUserId: _zod.z.string().optional(),
    changingStatusReason: _zod.z.string().optional().nullable().default(null)
}).strip();
medication.put('/medication-administration-record/:id/given', (0, _expressasynchandler.default)(async (req, res)=>{
    req.checkPermission('write', 'MedicationAdministration');
    const { models, params, db } = req;
    const { MedicationAdministrationRecord, MedicationAdministrationRecordDose, User } = models;
    const { dose, recordedByUserId, changingStatusReason } = await givenMarUpdateSchema.parseAsync(req.body);
    const record = await MedicationAdministrationRecord.findByPk(params.id);
    if (!record) {
        throw new _errors.InvalidOperationError(`MAR with id ${params.id} not found`);
    }
    if (record.status === _constants.ADMINISTRATION_STATUS.GIVEN) {
        throw new _errors.InvalidOperationError(`MAR with id ${params.id} is already given`);
    }
    if (recordedByUserId) {
        const recordedByUser = await User.findByPk(recordedByUserId);
        if (!recordedByUser) {
            throw new _errors.InvalidOperationError(`User with id ${recordedByUserId} not found`);
        }
    }
    if (dose.recordedByUserId) {
        const doseRecordedByUser = await User.findByPk(dose.recordedByUserId);
        if (!doseRecordedByUser) {
            throw new _errors.InvalidOperationError(`User with id ${dose.recordedByUserId} not found`);
        }
    }
    //validate givenByUserId
    if (dose.givenByUserId) {
        const givenByUser = await User.findByPk(dose.givenByUserId);
        if (!givenByUser) {
            throw new _errors.InvalidOperationError(`User with id ${dose.givenByUserId} not found`);
        }
    }
    const result = await db.transaction(async ()=>{
        record.status = _constants.ADMINISTRATION_STATUS.GIVEN;
        record.recordedByUserId = recordedByUserId || req.user.id;
        record.changingStatusReason = changingStatusReason;
        if (!record.recordedAt) {
            record.recordedAt = (0, _dateTime.getCurrentDateTimeString)();
        } else {
            record.isEdited = true;
        }
        await record.save();
        await MedicationAdministrationRecordDose.create({
            marId: record.id,
            doseAmount: dose.doseAmount,
            givenTime: dose.givenTime,
            givenByUserId: dose.givenByUserId || req.user.id,
            recordedByUserId: dose.recordedByUserId || req.user.id,
            doseIndex: 0
        });
        return record;
    });
    res.send(result.forResponse());
}));
const givenMarCreateSchema = _zod.z.object({
    dose: _zod.z.object({
        doseAmount: _zod.z.number(),
        givenTime: _dateTime.datetimeCustomValidation
    }),
    dueAt: _dateTime.datetimeCustomValidation,
    prescriptionId: _zod.z.string(),
    changingStatusReason: _zod.z.string().optional().nullable().default(null)
}).strip();
medication.post('/medication-administration-record/given', (0, _expressasynchandler.default)(async (req, res)=>{
    const { models, db } = req;
    const { MedicationAdministrationRecord, MedicationAdministrationRecordDose } = models;
    req.checkPermission('create', 'MedicationAdministration');
    const { dose, dueAt, prescriptionId, changingStatusReason } = await givenMarCreateSchema.parseAsync(req.body);
    //validate dose
    if (dose.doseAmount <= 0) {
        throw new _errors.InvalidOperationError(`Dose amount must be greater than 0`);
    }
    const result = await db.transaction(async ()=>{
        //create MAR
        const record = await MedicationAdministrationRecord.create({
            dueAt,
            prescriptionId,
            status: _constants.ADMINISTRATION_STATUS.GIVEN,
            recordedAt: (0, _dateTime.getCurrentDateTimeString)(),
            recordedByUserId: req.user.id,
            changingStatusReason
        });
        //create dose
        await MedicationAdministrationRecordDose.create({
            marId: record.id,
            doseAmount: dose.doseAmount,
            givenTime: dose.givenTime,
            givenByUserId: req.user.id,
            recordedByUserId: req.user.id,
            doseIndex: 0
        });
        return record;
    });
    res.send(result.forResponse());
}));
const notGivenMarInfoUpdateSchema = _zod.z.object({
    reasonNotGivenId: _zod.z.string(),
    recordedByUserId: _zod.z.string(),
    changingNotGivenInfoReason: _zod.z.string().optional().nullable().default(null)
}).strip();
medication.put('/medication-administration-record/:id/not-given-info', (0, _expressasynchandler.default)(async (req, res)=>{
    req.checkPermission('write', 'MedicationAdministration');
    const { models, params } = req;
    const { MedicationAdministrationRecord, User } = models;
    const { reasonNotGivenId, recordedByUserId, changingNotGivenInfoReason } = await notGivenMarInfoUpdateSchema.parseAsync(req.body);
    const record = await MedicationAdministrationRecord.findByPk(params.id);
    if (!record) {
        throw new _errors.InvalidOperationError(`MAR with id ${params.id} not found`);
    }
    const recordedByUser = await User.findByPk(recordedByUserId);
    if (!recordedByUser) {
        throw new _errors.InvalidOperationError(`User with id ${recordedByUserId} not found`);
    }
    const reasonNotGiven = await req.models.ReferenceData.findByPk(reasonNotGivenId, {
        where: {
            type: _constants.REFERENCE_TYPES.MEDICATION_NOT_GIVEN_REASON
        }
    });
    if (!reasonNotGiven) {
        throw new _errors.InvalidOperationError(`Not given reason with id ${reasonNotGivenId} not found`);
    }
    if (record.reasonNotGivenId === reasonNotGivenId && record.recordedByUserId === recordedByUserId) {
        throw new _errors.InvalidOperationError(`No changes were made to the MAR`);
    }
    record.reasonNotGivenId = reasonNotGivenId;
    record.recordedByUserId = recordedByUserId;
    record.changingNotGivenInfoReason = changingNotGivenInfoReason;
    record.isEdited = true;
    await record.save();
    res.send(record.forResponse());
}));
const notGivenMarUpdateSchema = _zod.z.object({
    reasonNotGivenId: _zod.z.string(),
    recordedByUserId: _zod.z.string().optional(),
    changingStatusReason: _zod.z.string().optional().nullable().default(null)
}).strip();
medication.put('/medication-administration-record/:id/not-given', (0, _expressasynchandler.default)(async (req, res)=>{
    req.checkPermission('write', 'MedicationAdministration');
    const { models, params, db } = req;
    const { MedicationAdministrationRecord, MedicationAdministrationRecordDose, User } = models;
    const { reasonNotGivenId, recordedByUserId, changingStatusReason } = await notGivenMarUpdateSchema.parseAsync(req.body);
    //validate not given reason
    const reasonNotGiven = await req.models.ReferenceData.findByPk(reasonNotGivenId, {
        where: {
            type: _constants.REFERENCE_TYPES.MEDICATION_NOT_GIVEN_REASON
        }
    });
    if (!reasonNotGiven) {
        throw new _errors.InvalidOperationError(`Not given reason with id ${reasonNotGivenId} not found`);
    }
    const record = await MedicationAdministrationRecord.findByPk(params.id);
    if (!record) {
        throw new _errors.InvalidOperationError(`MAR with id ${params.id} not found`);
    }
    if (record.status === _constants.ADMINISTRATION_STATUS.NOT_GIVEN) {
        throw new _errors.InvalidOperationError(`MAR with id ${params.id} is already not given`);
    }
    //validate recordedByUserId
    if (recordedByUserId) {
        const recordedByUser = await User.findByPk(recordedByUserId);
        if (!recordedByUser) {
            throw new _errors.InvalidOperationError(`User with id ${recordedByUserId} not found`);
        }
    }
    const result = await db.transaction(async ()=>{
        record.reasonNotGivenId = reasonNotGivenId;
        record.status = _constants.ADMINISTRATION_STATUS.NOT_GIVEN;
        record.recordedByUserId = recordedByUserId || req.user.id;
        record.changingStatusReason = changingStatusReason;
        if (!record.recordedAt) {
            record.recordedAt = (0, _dateTime.getCurrentDateTimeString)();
        } else {
            record.isEdited = true;
        }
        await record.save();
        await MedicationAdministrationRecordDose.destroy({
            where: {
                marId: record.id
            }
        });
        return record;
    });
    res.send(result.forResponse());
}));
const notGivenMarCreateSchema = _zod.z.object({
    reasonNotGivenId: _zod.z.string(),
    dueAt: _dateTime.datetimeCustomValidation,
    prescriptionId: _zod.z.string(),
    changingStatusReason: _zod.z.string().optional().nullable().default(null)
}).strip();
medication.post('/medication-administration-record/not-given', (0, _expressasynchandler.default)(async (req, res)=>{
    req.checkPermission('create', 'MedicationAdministration');
    const { models } = req;
    const { MedicationAdministrationRecord } = models;
    const { reasonNotGivenId, dueAt, prescriptionId, changingStatusReason } = await notGivenMarCreateSchema.parseAsync(req.body);
    //validate not given reason
    const reasonNotGiven = await req.models.ReferenceData.findByPk(reasonNotGivenId, {
        where: {
            type: _constants.REFERENCE_TYPES.MEDICATION_NOT_GIVEN_REASON
        }
    });
    if (!reasonNotGiven) {
        throw new _errors.InvalidOperationError(`Not given reason with id ${reasonNotGivenId} not found`);
    }
    //create MAR
    const record = await MedicationAdministrationRecord.create({
        reasonNotGivenId,
        dueAt,
        prescriptionId,
        status: _constants.ADMINISTRATION_STATUS.NOT_GIVEN,
        recordedAt: (0, _dateTime.getCurrentDateTimeString)(),
        recordedByUserId: req.user.id,
        changingStatusReason
    });
    res.send(record.forResponse());
}));
const updateMarSchema = _zod.z.object({
    isError: _zod.z.boolean().optional(),
    errorNotes: _zod.z.string().optional(),
    doses: _zod.z.array(_zod.z.object({
        doseAmount: _zod.z.number(),
        givenTime: _dateTime.datetimeCustomValidation,
        givenByUserId: _zod.z.string(),
        recordedByUserId: _zod.z.string()
    }).strip()).optional()
}).strip();
medication.put('/medication-administration-record/:id', (0, _expressasynchandler.default)(async (req, res)=>{
    req.checkPermission('write', 'MedicationAdministration');
    const { models, params, db } = req;
    const marId = params.id;
    const { MedicationAdministrationRecord, MedicationAdministrationRecordDose, User, Prescription, Note } = models;
    const { isError, errorNotes, doses } = await updateMarSchema.parseAsync(req.body);
    const existingMar = await MedicationAdministrationRecord.findByPk(marId, {
        include: [
            {
                model: Prescription,
                as: 'prescription',
                include: [
                    'encounterPrescription',
                    'medication'
                ]
            }
        ]
    });
    if (!existingMar) {
        throw new _errors.InvalidOperationError(`MAR with id ${marId} not found`);
    }
    if (existingMar.status !== _constants.ADMINISTRATION_STATUS.GIVEN && doses?.length) {
        throw new _errors.InvalidOperationError(`MAR with id ${marId} is not given and cannot have doses`);
    }
    const result = await db.transaction(async ()=>{
        if (isError) {
            existingMar.isError = isError;
            existingMar.errorNotes = errorNotes;
            await Note.create({
                content: `Medication error recorded for ${existingMar.prescription.medication.name} dose recorded at ${existingMar.recordedAt}. ${errorNotes || ''}`.trim(),
                authorId: req.user.id,
                recordId: existingMar.prescription.encounterPrescription.encounterId,
                date: (0, _dateTime.getCurrentDateTimeString)(),
                noteType: _constants.NOTE_TYPES.SYSTEM,
                recordType: _constants.NOTE_RECORD_TYPES.ENCOUNTER
            });
            await existingMar.save();
        }
        if (doses.length) {
            for (const dose of doses){
                const givenByUser = await User.findByPk(dose.givenByUserId);
                if (!givenByUser) {
                    throw new _errors.InvalidOperationError(`User with id ${dose.givenByUserId} not found`);
                }
                const recordedByUser = await User.findByPk(dose.recordedByUserId);
                if (!recordedByUser) {
                    throw new _errors.InvalidOperationError(`User with id ${dose.recordedByUserId} not found`);
                }
            }
            const lastDoseIndex = await MedicationAdministrationRecordDose.max('doseIndex', {
                where: {
                    marId
                }
            });
            const dosesToCreate = doses.map((d, index)=>({
                    ...d,
                    marId,
                    doseIndex: lastDoseIndex + index + 1
                }));
            await MedicationAdministrationRecordDose.bulkCreate(dosesToCreate);
        }
        return existingMar;
    });
    res.send(result.forResponse());
}));
medication.get('/medication-administration-record/:id/doses', (0, _expressasynchandler.default)(async (req, res)=>{
    req.checkPermission('read', 'MedicationAdministration');
    const { models, params } = req;
    const { MedicationAdministrationRecordDose } = models;
    const doses = await MedicationAdministrationRecordDose.findAll({
        where: {
            marId: params.id
        },
        include: [
            {
                association: 'givenByUser',
                attributes: [
                    'id',
                    'displayName'
                ]
            },
            {
                association: 'recordedByUser',
                attributes: [
                    'id',
                    'displayName'
                ]
            }
        ],
        order: [
            [
                'doseIndex',
                'ASC'
            ]
        ]
    });
    res.send({
        count: doses.length,
        data: doses.map((dose)=>dose.forResponse())
    });
}));
const updateDoseSchema = _zod.z.object({
    doseAmount: _zod.z.number(),
    givenTime: _dateTime.datetimeCustomValidation,
    givenByUserId: _zod.z.string(),
    recordedByUserId: _zod.z.string(),
    reasonForChange: _zod.z.string().optional().nullable().default(null)
});
medication.put('/medication-administration-record/doses/:doseId', (0, _expressasynchandler.default)(async (req, res)=>{
    req.checkPermission('write', 'MedicationAdministration');
    const { models, params, db } = req;
    const { doseId } = params;
    const { MedicationAdministrationRecordDose, User, MedicationAdministrationRecord } = models;
    const { doseAmount, givenTime, givenByUserId, recordedByUserId, reasonForChange } = await updateDoseSchema.parseAsync(req.body);
    const doseObject = await MedicationAdministrationRecordDose.findByPk(doseId);
    if (!doseObject) {
        throw new _errors.InvalidOperationError(`Dose with id ${doseId} not found`);
    }
    const givenByUser = await User.findByPk(givenByUserId);
    if (!givenByUser) {
        throw new _errors.InvalidOperationError(`User with id ${givenByUserId} not found`);
    }
    const recordedByUser = await User.findByPk(recordedByUserId);
    if (!recordedByUser) {
        throw new _errors.InvalidOperationError(`User with id ${recordedByUserId} not found`);
    }
    if (Number(doseObject.doseAmount) === doseAmount && (0, _datefns.isEqual)(new Date(doseObject.givenTime), new Date(givenTime)) && doseObject.givenByUserId === givenByUserId && doseObject.recordedByUserId === recordedByUserId) {
        throw new _errors.InvalidOperationError(`No changes were made to the dose`);
    }
    const marId = doseObject.marId;
    const result = await db.transaction(async ()=>{
        await MedicationAdministrationRecord.update({
            isEdited: true
        }, {
            where: {
                id: marId
            }
        });
        doseObject.doseAmount = doseAmount;
        doseObject.givenTime = givenTime;
        doseObject.givenByUserId = givenByUserId;
        doseObject.recordedByUserId = recordedByUserId;
        doseObject.reasonForChange = reasonForChange;
        await doseObject.save();
        return doseObject;
    });
    res.send(result.forResponse());
}));
const deleteDoseInputSchema = _zod.z.object({
    reasonForRemoval: _zod.z.string().optional().nullable().default(null)
}).strip();
medication.delete('/medication-administration-record/doses/:doseId', (0, _expressasynchandler.default)(async (req, res)=>{
    req.checkPermission('write', 'MedicationAdministration');
    const { models, params, query } = req;
    const doseId = params.doseId;
    const { MedicationAdministrationRecordDose } = models;
    const { reasonForRemoval } = await deleteDoseInputSchema.parseAsync(query);
    const existingDose = await MedicationAdministrationRecordDose.findByPk(doseId);
    if (!existingDose) {
        throw new _errors.InvalidOperationError(`Dose with id ${doseId} not found`);
    }
    if (existingDose.doseIndex === 0) {
        throw new _errors.InvalidOperationError(`Cannot delete primary dose`);
    }
    existingDose.reasonForRemoval = reasonForRemoval;
    existingDose.isRemoved = true;
    await existingDose.save();
    res.send(existingDose.forResponse());
}));
medication.get('/medication-administration-record/:id/changelog', (0, _expressasynchandler.default)(async (req, res)=>{
    req.checkPermission('read', 'MedicationAdministration');
    const { models, params, db } = req;
    const { MedicationAdministrationRecord } = models;
    const marId = params.id;
    // Verify the MAR exists
    const mar = await MedicationAdministrationRecord.findByPk(marId);
    if (!mar) {
        throw new _errors.InvalidOperationError(`MAR with id ${marId} not found`);
    }
    // Use raw SQL query to get changelog data
    const query = `
      select
        case
          when c.table_name = 'medication_administration_records' then 'mar'
          when c.table_name = 'medication_administration_record_doses' then 'dose'
        end as type,
        c.record_data->>'id' id,
        c.record_data->>'status' mar_status,
        c.record_data->>'changing_status_reason' mar_changing_status_reason,
        c.record_data->>'changing_not_given_info_reason' mar_changing_not_given_info_reason,
        reason_not_given.name mar_not_given_reason_name,
        reason_not_given.id mar_not_given_reason_id,

        c.record_data->>'dose_index' dose_index,
        c.record_data->>'dose_amount' dose_amount,
        c.record_data->>'given_time' dose_given_time,
        c.record_data->>'is_removed' dose_is_removed,
        c.record_data->>'reason_for_removal' dose_reason_for_removal,
        c.record_data->>'reason_for_change' dose_reason_for_change,
        given_by_user.display_name dose_given_user_name,
        given_by_user.id dose_given_user_id,

        recorded_by_user.display_name recorded_by_user_name,
        recorded_by_user.id recorded_by_user_id,

        c.created_at,
        c.record_created_at record_created_at,
        c.record_updated_at record_updated_at,
        c.record_deleted_at record_deleted_at,
        updated_by_user.display_name as changed_by_user
      from logs.changes c
      left join public.users updated_by_user on updated_by_user.id = c.updated_by_user_id
      left join public.reference_data reason_not_given on reason_not_given.type = :medicationNotGivenReason and reason_not_given.id = c.record_data->>'reason_not_given_id'
      left join public.users recorded_by_user on recorded_by_user.id = c.record_data->>'recorded_by_user_id'
      left join public.users given_by_user on given_by_user.id = c.record_data->>'given_by_user_id'
      where c.table_schema = 'public'
      and c.table_name IN ('medication_administration_records', 'medication_administration_record_doses')
      and c.migration_context IS NULL
      and (c.record_data->>'id' = :marId or c.record_data->>'mar_id' = :marId)
      order by c.created_at desc,
        case
          when c.table_name = 'medication_administration_records' then 1
          when c.table_name = 'medication_administration_record_doses' then 2
        end desc,
        c.record_data->>'dose_index' desc
    `;
    const results = await db.query(query, {
        replacements: {
            marId,
            medicationNotGivenReason: _constants.REFERENCE_TYPES.MEDICATION_NOT_GIVEN_REASON
        },
        type: _sequelize.QueryTypes.SELECT
    });
    // Transform results for the response
    const transformedResults = results.map((result)=>({
            type: result.type,
            id: result.id,
            marStatus: result.mar_status,
            marChangingStatusReason: result.mar_changing_status_reason,
            marChangingNotGivenInfoReason: result.mar_changing_not_given_info_reason,
            marNotGivenReason: {
                id: result.mar_not_given_reason_id,
                name: result.mar_not_given_reason_name
            },
            doseIndex: result.type === 'dose' ? Number(result.dose_index) : null,
            doseAmount: result.dose_amount,
            doseGivenTime: result.dose_given_time,
            doseIsRemoved: result.dose_is_removed,
            doseReasonForRemoval: result.dose_reason_for_removal,
            doseReasonForChange: result.dose_reason_for_change,
            doseGivenByUser: {
                id: result.dose_given_user_id,
                name: result.dose_given_user_name
            },
            recordedByUser: {
                id: result.recorded_by_user_id,
                name: result.recorded_by_user_name
            },
            changedByUser: result.changed_by_user,
            createdAt: result.created_at,
            recordCreatedAt: result.record_created_at,
            recordUpdatedAt: result.record_updated_at,
            recordDeletedAt: result.record_deleted_at,
            changeType: new Date(result.record_created_at).getTime() === new Date(result.record_updated_at).getTime() ? 'CREATED' : 'UPDATED'
        }));
    res.send(transformedResults);
}));

//# sourceMappingURL=medication.js.map