"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
function _export(target, all) {
    for(var name in all)Object.defineProperty(target, name, {
        enumerable: true,
        get: all[name]
    });
}
_export(exports, {
    provision: function() {
        return provision;
    },
    provisionCommand: function() {
        return provisionCommand;
    }
});
const _nodepath = require("node:path");
const _commander = require("commander");
const _lodash = require("lodash");
const _sequelize = require("sequelize");
const _xlsx = require("xlsx");
const _constants = require("@tamanu/constants");
const _logging = require("@tamanu/shared/services/logging");
const _database = require("../database");
const _integrations = require("../integrations");
const _loadSettingFile = require("../utils/loadSettingFile");
const _referenceDataImporter = require("../admin/referenceDataImporter");
const _importerEndpoint = require("../admin/importer/importerEndpoint");
const _utils = require("../auth/utils");
const _programImporter = require("../admin/programImporter/programImporter");
/**
 * Validates that a reference data file contains all sheets importable through the reference data importer
 * @param {string} file - File path
 */ function validateFullReferenceDataImport(file) {
    // These are two very unique cases. 'user' has special logic and 'administeredVaccine' is a special case used for existing deployments.
    const EXCLUDED_FROM_FULL_IMPORT_CHECK = [
        'user',
        'administeredVaccine'
    ];
    _logging.log.debug('Parse XLSX workbook for validation');
    const workbook = (0, _xlsx.readFile)(file);
    const sheetNameDictionary = (0, _lodash.keyBy)(Object.keys(workbook.Sheets), _importerEndpoint.normaliseSheetName);
    // Check all required data types are present and have data
    const missingDataTypes = [];
    for (const dataType of _constants.GENERAL_IMPORTABLE_DATA_TYPES){
        if (EXCLUDED_FROM_FULL_IMPORT_CHECK.includes(dataType)) continue;
        const sheetName = sheetNameDictionary[dataType];
        if (!sheetName || _xlsx.utils.sheet_to_json(workbook.Sheets[sheetName]).length === 0) {
            missingDataTypes.push(dataType);
        }
    }
    if (missingDataTypes.length > 0) {
        throw new Error(`Reference data file has no rows for the following data types:\n${missingDataTypes.join('\n')}`);
    }
    _logging.log.info('Reference data file validation passed - all required sheets contain data');
}
async function provision(provisioningFile, { skipIfNotNeeded }) {
    const store = await (0, _database.initDatabase)({
        testMode: false
    });
    const userCount = await store.models.User.count({
        where: {
            id: {
                [_sequelize.Op.ne]: _constants.SYSTEM_USER_UUID
            }
        }
    });
    if (userCount > 0) {
        if (skipIfNotNeeded) {
            _logging.log.info(`Found ${userCount} users already in the database, but expecting to, not provisioning`);
            return;
        }
        throw new Error(`Found ${userCount} users already in the database, aborting provision`);
    }
    (0, _integrations.checkIntegrationsConfig)();
    const { users = {}, facilities = {}, programs = [], referenceData = [], settings = {} } = await (0, _loadSettingFile.loadSettingFile)(provisioningFile);
    /// //////////////
    /// REFERENCE DATA
    const errors = [];
    const stats = [];
    const importerOptions = {
        errors,
        models: store.models,
        stats,
        includedDataTypes: [
            ..._constants.GENERAL_IMPORTABLE_DATA_TYPES,
            ..._constants.PERMISSION_IMPORTABLE_DATA_TYPES
        ],
        checkPermission: ()=>true
    };
    for (const { file: referenceDataFile = null, url: referenceDataUrl = null, defaultSpreadsheet: isUsingDefaultSpreadsheet = false, ...rest } of referenceData ?? []){
        if (isUsingDefaultSpreadsheet) {
            const defaultReferenceDataFile = (0, _nodepath.resolve)(__dirname, 'default-provisioning.xlsx');
            _logging.log.info('Using reference data spreadsheet from this branch', {
                file: defaultReferenceDataFile
            });
            // We only validate the default import to ensure it stays complete. It is fine to allow partial imports through the other options.
            validateFullReferenceDataImport(defaultReferenceDataFile);
            await (0, _referenceDataImporter.referenceDataImporter)({
                file: defaultReferenceDataFile,
                ...importerOptions
            });
        }
        if (!referenceDataFile && !referenceDataUrl && !isUsingDefaultSpreadsheet) {
            throw new Error(`Unknown reference data import with keys ${Object.keys(rest).join(', ')}`);
        }
        if (referenceDataFile) {
            const realpath = (0, _nodepath.resolve)(provisioningFile, referenceDataFile);
            _logging.log.info('Importing reference data file', {
                file: realpath
            });
            await (0, _referenceDataImporter.referenceDataImporter)({
                file: realpath,
                ...importerOptions
            });
        } else if (referenceDataUrl) {
            _logging.log.info('Downloading reference data file', {
                url: referenceDataUrl
            });
            const file = await fetch(referenceDataUrl);
            const data = Buffer.from(await (await file.blob()).arrayBuffer());
            _logging.log.info('Importing reference data', {
                size: data.byteLength
            });
            await (0, _referenceDataImporter.referenceDataImporter)({
                data,
                file: referenceDataUrl,
                ...importerOptions
            });
        }
    }
    if (errors.length) {
        for (const error of errors){
            _logging.log.error(error);
        }
        throw new Error(`Encountered ${errors.length} errors during provisioning`);
    }
    _logging.log.info('Imported reference data successfully', stats);
    /// //////////
    /// FACILITIES
    for (const [id, value] of Object.entries(facilities)){
        const fields = {
            ...value
        };
        delete fields.user;
        delete fields.password;
        delete fields.settings;
        const facility = await store.models.Facility.findByPk(id);
        if (facility) {
            _logging.log.info('Updating facility', {
                id
            });
            await facility.update(fields);
        } else {
            var _fields, _fields1;
            _logging.log.info('Creating facility', {
                id
            });
            (_fields = fields).name || (_fields.name = id);
            (_fields1 = fields).code || (_fields1.code = id);
            await store.models.Facility.create({
                id,
                ...fields
            });
        }
    }
    /// ////////
    /// SETTINGS
    const combineSettings = async (settingData, scope, facilityId)=>{
        const existing = await store.models.Setting.get('', facilityId, scope);
        const combined = (0, _lodash.defaultsDeep)(settingData, existing);
        return store.models.Setting.set('', combined, scope, facilityId);
    };
    if (settings.global) {
        await combineSettings(settings.global, _constants.SETTINGS_SCOPES.GLOBAL);
        _logging.log.info('Set global settings');
    }
    if (settings.facilities) {
        await Promise.all(Object.entries(settings.facilities).map(([facilityId, facilitySettings])=>combineSettings(facilitySettings, _constants.SETTINGS_SCOPES.FACILITY, facilityId)));
        _logging.log.info('Set facility settings');
    }
    if (settings.central) {
        await combineSettings(settings.central, _constants.SETTINGS_SCOPES.CENTRAL);
        _logging.log.info('Set central settings');
    }
    /// /////
    /// USERS
    const allUsers = [
        ...Object.entries(users),
        ...Object.entries(facilities).map(([id, { user, password }])=>user && password && [
                user,
                {
                    displayName: `System: ${id} sync`,
                    password
                }
            ]).filter(Boolean)
    ];
    for (const [email, { role = 'admin', password, ...fields }] of allUsers){
        let realPassword = password;
        if (!realPassword) {
            realPassword = (0, _utils.getRandomBase64String)(16);
            // eslint-disable-next-line no-console
            console.log(`NEW PASSWORD for ${email}: ${realPassword}`);
        }
        const user = await store.models.User.findOne({
            where: {
                email
            }
        });
        if (user) {
            _logging.log.info('Updating user', {
                email
            });
            user.set({
                role,
                ...fields
            });
            user.setPassword(realPassword);
            await user.save();
        } else {
            _logging.log.info('Creating user', {
                email
            });
            await store.models.User.create({
                email,
                role,
                password: realPassword,
                ...fields
            });
        }
    }
    /// ////////
    /// PROGRAMS
    const programOptions = {
        errors,
        models: store.models,
        stats,
        checkPermission: ()=>true
    };
    for (const { file: programFile = null, url: programUrl = null, ...rest } of programs){
        if (!programFile && !programUrl) {
            throw new Error(`Unknown program import with keys ${Object.keys(rest).join(', ')}`);
        }
        if (programFile) {
            const realpath = (0, _nodepath.resolve)(provisioningFile, programFile);
            _logging.log.info('Importing program file', {
                file: realpath
            });
            await (0, _programImporter.programImporter)({
                file: realpath,
                ...programOptions
            });
        } else if (programUrl) {
            _logging.log.info('Downloading program file', {
                url: programUrl
            });
            const file = await fetch(programUrl);
            const data = Buffer.from(await (await file.blob()).arrayBuffer());
            _logging.log.info('Importing program', {
                size: data.byteLength
            });
            await (0, _programImporter.programImporter)({
                data,
                file: programUrl,
                ...programOptions
            });
        }
    }
    if (errors.length) {
        for (const error of errors){
            _logging.log.error(error);
        }
        throw new Error(`Encountered ${errors.length} errors during provisioning`);
    }
    _logging.log.info('Imported programs successfully', stats);
    _logging.log.info('Done.');
}
const provisionCommand = new _commander.Command('provision').description('Set up initial data. See https://beyond-essential.slab.com/posts/tamanu-provisioning-file-h1urgi86 for details or /docs/provisioning/example.json5 for a sample file.').argument('<file>', 'Path to the provisioning file').option('--skip-if-not-needed', 'If there are already users in the database, exit(0) instead of aborting').action(provision);

//# sourceMappingURL=provision.js.map