"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "GenerateRepeatingTasks", {
    enumerable: true,
    get: function() {
        return GenerateRepeatingTasks;
    }
});
const _config = /*#__PURE__*/ _interop_require_default(require("config"));
const _tasks = require("@tamanu/shared/tasks");
const _logging = require("@tamanu/shared/services/logging");
const _sequelize = require("sequelize");
const _sleepAsync = require("@tamanu/utils/sleepAsync");
const _ = require(".");
const _constants = require("@tamanu/constants");
const _dateTime = require("@tamanu/utils/dateTime");
function _interop_require_default(obj) {
    return obj && obj.__esModule ? obj : {
        default: obj
    };
}
const MILLISECONDS_PER_DAY = 86400000;
let GenerateRepeatingTasks = class GenerateRepeatingTasks extends _tasks.ScheduledTask {
    getName() {
        return 'GenerateRepeatingTasks';
    }
    async run() {
        await this.removeChildTasksOverParentEndTime();
        await this.markOldRepeatingTasksAsNotCompleted();
        await this.generateChildTasks();
    }
    // remove child tasks that have dueTime over parent endTime
    // this is a safe guard for the delayed sync from facility server to central server
    // only need to account for tasks that have been created in the last 30 days with the status of TODO
    async removeChildTasksOverParentEndTime() {
        try {
            await this.sequelize.query(`
      WITH parent as (
        select id, deleted_reason_for_sync_id as reason_id
        FROM tasks
      )
      UPDATE tasks
      SET
        deleted_at = now (),
        deleted_by_user_id = :deletedByUserId,
        deleted_time = :deletedTime,
        deleted_reason_id = parent.reason_id
      FROM parent
      WHERE
        tasks.parent_task_id = parent.id AND
        tasks.id in (
          SELECT
            childTasks.id
          FROM
            tasks as childTasks
            JOIN tasks as parent_tasks ON parent_tasks.id = childTasks.parent_task_id
            AND parent_tasks.end_time IS NOT NULL
          WHERE
            childTasks.deleted_at is NULL
            and childTasks.due_time > parent_tasks.end_time
            and childTasks.status IN (:statuses)
            and childTasks.created_at > now () - interval '30' day
        )
    `, {
                replacements: {
                    statuses: [
                        _constants.TASK_STATUSES.TODO
                    ],
                    deletedByUserId: _constants.SYSTEM_USER_UUID,
                    deletedTime: (0, _dateTime.getCurrentDateTimeString)()
                }
            });
        } catch (error) {
            console.error(error);
        }
    }
    async markOldRepeatingTasksAsNotCompleted() {
        const { Task, ReferenceData } = this.models;
        const notCompletedReason = await ReferenceData.findByPk(_constants.TASK_NOTE_COMPLETE_OVERDUE_REASON_ID);
        // 2 days ago
        const cutoffDateTime = new Date(new Date().getTime() - 2 * MILLISECONDS_PER_DAY);
        await Task.update({
            status: _constants.TASK_STATUSES.NON_COMPLETED,
            notCompletedByUserId: _constants.SYSTEM_USER_UUID,
            notCompletedTime: (0, _dateTime.getCurrentDateTimeString)(),
            notCompletedReasonId: notCompletedReason?.id || null
        }, {
            where: {
                status: _constants.TASK_STATUSES.TODO,
                frequencyValue: {
                    [_sequelize.Op.not]: null
                },
                frequencyUnit: {
                    [_sequelize.Op.not]: null
                },
                dueTime: {
                    [_sequelize.Op.lt]: (0, _dateTime.toDateTimeString)(cutoffDateTime)
                }
            }
        });
    }
    //generate child tasks for repeating tasks
    async generateChildTasks() {
        const { Task } = this.models;
        const toProcess = await Task.count({
            endTime: null,
            frequencyValue: {
                [_sequelize.Op.not]: null
            },
            frequencyUnit: {
                [_sequelize.Op.not]: null
            },
            parentTaskId: null
        });
        if (toProcess === 0) return;
        const { batchSize, batchSleepAsyncDurationInMilliseconds } = this.config;
        if (!batchSize || !batchSleepAsyncDurationInMilliseconds) {
            throw new _.InvalidConfigError('batchSize and batchSleepAsyncDurationInMilliseconds must be set for GenerateRepeatingTasks');
        }
        const batchCount = Math.ceil(toProcess / batchSize);
        _logging.log.info('Running batched generating repeating tasks', {
            recordCount: toProcess,
            batchCount,
            batchSize
        });
        for(let i = 0; i < batchCount; i++){
            const tasks = await Task.findAll({
                where: {
                    endTime: null,
                    frequencyValue: {
                        [_sequelize.Op.not]: null
                    },
                    frequencyUnit: {
                        [_sequelize.Op.not]: null
                    },
                    parentTaskId: null
                },
                include: [
                    'designations'
                ],
                limit: batchSize
            });
            await Task.generateRepeatingTasks(tasks);
            await (0, _sleepAsync.sleepAsync)(batchSleepAsyncDurationInMilliseconds);
        }
    }
    /**
   *
   * @param {import('../ApplicationContext').ApplicationContext} context
   */ constructor(context){
        const conf = _config.default.schedules.generateRepeatingTasks;
        const { schedule, jitterTime, enabled } = conf;
        super(schedule, _logging.log, jitterTime, enabled);
        this.models = context.store.models;
        this.config = conf;
        this.sequelize = context.store.sequelize;
    }
};

//# sourceMappingURL=GenerateRepeatingTasks.js.map