"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "UserLoginAttempt", {
    enumerable: true,
    get: function() {
        return UserLoginAttempt;
    }
});
const _constants = require("@tamanu/constants");
const _Model = require("./Model");
const _sequelize = require("sequelize");
let UserLoginAttempt = class UserLoginAttempt extends _Model.Model {
    static initModel(options) {
        super.init({
            id: {
                type: _sequelize.DataTypes.UUID,
                allowNull: false,
                primaryKey: true,
                defaultValue: _sequelize.Sequelize.fn('gen_random_uuid')
            },
            outcome: {
                type: _sequelize.DataTypes.TEXT,
                allowNull: false
            }
        }, {
            ...options,
            syncDirection: _constants.SYNC_DIRECTIONS.PUSH_TO_CENTRAL
        });
    }
    static initRelations(models) {
        this.belongsTo(models.User, {
            foreignKey: 'userId',
            as: 'user'
        });
        this.belongsTo(models.Device, {
            foreignKey: 'deviceId',
            as: 'device'
        });
    }
    static async checkIsUserLockedOut({ settings, userId, deviceId = '' }) {
        const { lockoutDuration } = await settings.get(_constants.SETTING_KEYS.SECURITY_LOGIN_ATTEMPTS);
        const lockedAttempt = await this.findOne({
            where: {
                userId,
                deviceId,
                outcome: _constants.LOGIN_ATTEMPT_OUTCOMES.LOCKED,
                createdAt: {
                    [_sequelize.Op.gte]: _sequelize.Sequelize.literal("CURRENT_TIMESTAMP - $lockoutDuration * interval '1 minute'")
                }
            },
            order: [
                [
                    'createdAt',
                    'DESC'
                ]
            ],
            bind: {
                lockoutDuration
            }
        });
        const createdAt = lockedAttempt?.createdAt?.getTime() ?? 0;
        const remainingLockout = lockoutDuration * 60 - Math.floor((Date.now() - createdAt) / 1000);
        return {
            isUserLockedOut: !!lockedAttempt,
            remainingLockout: Math.max(0, remainingLockout)
        };
    }
    static async createFailedLoginAttempt({ settings, userId, deviceId = '' }) {
        const { lockoutThreshold, observationWindow, lockoutDuration } = await settings.get(_constants.SETTING_KEYS.SECURITY_LOGIN_ATTEMPTS);
        const failedAttempts = await this.count({
            where: {
                userId,
                deviceId,
                outcome: _constants.LOGIN_ATTEMPT_OUTCOMES.FAILED,
                createdAt: {
                    [_sequelize.Op.gte]: _sequelize.Sequelize.literal("CURRENT_TIMESTAMP - $observationWindow * interval '1 minute'")
                }
            },
            // @ts-ignore - sequelize doesn't know bind works in count
            bind: {
                observationWindow
            }
        });
        // We need to add 1 to the failed attempts because the current attempt is not included in the count
        const outcome = failedAttempts + 1 >= lockoutThreshold ? _constants.LOGIN_ATTEMPT_OUTCOMES.LOCKED : _constants.LOGIN_ATTEMPT_OUTCOMES.FAILED;
        const loginAttempt = await this.create({
            userId,
            deviceId,
            outcome
        });
        return {
            loginAttempt,
            lockoutDuration: lockoutDuration * 60,
            remainingAttempts: Math.max(0, lockoutThreshold - failedAttempts - 1)
        };
    }
};

//# sourceMappingURL=UserLoginAttempt.js.map