"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "createApi", {
    enumerable: true,
    get: function() {
        return createApi;
    }
});
const _bodyparser = /*#__PURE__*/ _interop_require_default(require("body-parser"));
const _compression = /*#__PURE__*/ _interop_require_default(require("compression"));
const _config = /*#__PURE__*/ _interop_require_default(require("config"));
const _express = /*#__PURE__*/ _interop_require_default(require("express"));
const _expressasynchandler = /*#__PURE__*/ _interop_require_default(require("express-async-handler"));
const _helmet = /*#__PURE__*/ _interop_require_default(require("helmet"));
const _logging = require("@tamanu/shared/services/logging");
const _middleware = require("@tamanu/shared/permissions/middleware");
const _constants = require("@tamanu/constants");
const _buildRoutes = require("./buildRoutes");
const _auth = require("./auth");
const _publicRoutes = require("./publicRoutes");
const _patientPortalApi = require("./patientPortalApi");
const _errorHandler = require("./middleware/errorHandler");
const _loadshedder = require("./middleware/loadshedder");
const _versionCompatibility = require("./middleware/versionCompatibility");
const _serverInfo = require("./serverInfo");
const _translation = require("./translation");
const _http = require("http");
const _middleware1 = require("@tamanu/settings/middleware");
const _audit = require("@tamanu/database/utils/audit");
function _interop_require_default(obj) {
    return obj && obj.__esModule ? obj : {
        default: obj
    };
}
function _getRequireWildcardCache(nodeInterop) {
    if (typeof WeakMap !== "function") return null;
    var cacheBabelInterop = new WeakMap();
    var cacheNodeInterop = new WeakMap();
    return (_getRequireWildcardCache = function(nodeInterop) {
        return nodeInterop ? cacheNodeInterop : cacheBabelInterop;
    })(nodeInterop);
}
function _interop_require_wildcard(obj, nodeInterop) {
    if (!nodeInterop && obj && obj.__esModule) {
        return obj;
    }
    if (obj === null || typeof obj !== "object" && typeof obj !== "function") {
        return {
            default: obj
        };
    }
    var cache = _getRequireWildcardCache(nodeInterop);
    if (cache && cache.has(obj)) {
        return cache.get(obj);
    }
    var newObj = {
        __proto__: null
    };
    var hasPropertyDescriptor = Object.defineProperty && Object.getOwnPropertyDescriptor;
    for(var key in obj){
        if (key !== "default" && Object.prototype.hasOwnProperty.call(obj, key)) {
            var desc = hasPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : null;
            if (desc && (desc.get || desc.set)) {
                Object.defineProperty(newObj, key, desc);
            } else {
                newObj[key] = obj[key];
            }
        }
    }
    newObj.default = obj;
    if (cache) {
        cache.set(obj, newObj);
    }
    return newObj;
}
const rawBodySaver = function(req, res, buf) {
    if (buf && buf.length) {
        req.rawBody = buf;
    }
};
function api(ctx) {
    const apiRoutes = _express.default.Router();
    apiRoutes.use('/public', _publicRoutes.publicRoutes);
    apiRoutes.post('/timesync', _bodyparser.default.raw({
        verify: rawBodySaver,
        type: '*/*'
    }), (0, _expressasynchandler.default)(async (req, res)=>{
        try {
            const timeRes = await ctx.timesync.answerClient(req.rawBody);
            res.type('application/octet-stream');
            res.send(timeRes);
        } catch (error) {
            _logging.log.error('Error in timesync:', error);
            res.status(500).send(error.toString());
        }
    }));
    apiRoutes.use(_auth.authModule);
    apiRoutes.use(_audit.attachAuditUserToDbSession);
    apiRoutes.use('/translation', _translation.translationRoutes);
    apiRoutes.use(_middleware.constructPermission);
    apiRoutes.use((0, _buildRoutes.buildRoutes)(ctx));
    return apiRoutes;
}
async function createApi(ctx) {
    const { store, emailService, reportSchemaStores } = ctx;
    const express = (0, _express.default)();
    let errorMiddleware = null;
    if (_config.default.errors?.enabled) {
        if (_config.default.errors?.type === 'bugsnag') {
            const { default: Bugsnag } = await Promise.resolve().then(()=>/*#__PURE__*/ _interop_require_wildcard(require("@bugsnag/js")));
            const middleware = Bugsnag.getPlugin('express');
            express.use(middleware.requestHandler);
            errorMiddleware = middleware.errorHandler;
        }
    }
    express.use((0, _helmet.default)({
        crossOriginEmbedderPolicy: true,
        strictTransportSecurity: false
    }));
    express.use((0, _loadshedder.loadshedder)());
    express.use((0, _compression.default)());
    express.use(_bodyparser.default.json({
        verify: rawBodySaver,
        limit: '50mb'
    }));
    express.use(_bodyparser.default.urlencoded({
        verify: rawBodySaver,
        extended: true
    }));
    // trust the x-forwarded-for header from addresses in `config.proxy.trusted`
    express.set('trust proxy', _config.default.proxy.trusted);
    express.use((0, _logging.getLoggingMiddleware)());
    express.use((req, res, next)=>{
        res.setHeader('X-Tamanu-Server', _constants.SERVER_TYPES.CENTRAL);
        res.setHeader('X-Version', _serverInfo.version);
        next();
    });
    express.use(_versionCompatibility.versionCompatibility);
    express.use((req, res, next)=>{
        req.models = store.models; // cross-compatibility with facility for shared middleware
        req.db = store.sequelize;
        req.store = store;
        req.emailService = emailService;
        req.reportSchemaStores = reportSchemaStores;
        req.ctx = ctx;
        req.language = req.headers['language'];
        next();
    });
    express.use(_middleware1.settingsReaderMiddleware);
    express.get('/$', (req, res)=>{
        res.send({
            index: true
        });
    });
    // Patient Portal - must go before main API to avoid main authentication
    express.use('/api/portal', async (req, res, next)=>{
        const patientPortalEnabled = await req.settings.get('features.patientPortal');
        return patientPortalEnabled ? (0, _patientPortalApi.patientPortalApi)(req, res, next) : res.status(501).end();
    });
    // API
    express.use('/api', api(ctx));
    // Legacy API endpoint
    express.use('/v1', api(ctx));
    // Dis-allow all other routes
    express.use('*', (req, res)=>{
        res.status(404).end();
    });
    if (errorMiddleware) {
        express.use(errorMiddleware);
    }
    express.use(_errorHandler.defaultErrorHandler);
    return {
        express,
        httpServer: (0, _http.createServer)(express)
    };
}

//# sourceMappingURL=createApi.js.map