"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "InvoicePriceList", {
    enumerable: true,
    get: function() {
        return InvoicePriceList;
    }
});
const _sequelize = require("sequelize");
const _constants = require("@tamanu/constants");
const _Model = require("../Model");
const _invoicePriceListMatching = require("./invoicePriceListMatching");
let InvoicePriceList = class InvoicePriceList extends _Model.Model {
    static initModel({ primaryKey, ...options }) {
        super.init({
            id: primaryKey,
            code: {
                type: _sequelize.DataTypes.STRING,
                allowNull: false
            },
            name: {
                type: _sequelize.DataTypes.STRING,
                allowNull: true
            },
            rules: {
                type: _sequelize.DataTypes.JSONB,
                allowNull: true
            },
            visibilityStatus: {
                type: _sequelize.DataTypes.TEXT,
                defaultValue: _constants.VISIBILITY_STATUSES.CURRENT,
                allowNull: false
            }
        }, {
            ...options,
            syncDirection: _constants.SYNC_DIRECTIONS.PULL_FROM_CENTRAL,
            indexes: [
                {
                    unique: true,
                    fields: [
                        'code'
                    ]
                }
            ]
        });
    }
    static initRelations(models) {
        this.hasMany(models.InvoicePriceListItem, {
            foreignKey: 'invoicePriceListId',
            as: 'invoicePriceListItems'
        });
    }
    static buildSyncFilter() {
        return null; // syncs everywhere
    }
    static async buildSyncLookupQueryDetails() {
        return null; // syncs everywhere
    }
    // Returns the id of the PriceList whose rules match the provided inputs
    // Throws an error if more than one match is found
    static async getIdForPatientEncounter(encounterId) {
        const { models } = this.sequelize;
        const encounter = await models.Encounter.findByPk(encounterId, {
            include: [
                {
                    association: 'patient',
                    include: [
                        {
                            association: 'additionalData'
                        }
                    ]
                },
                'location'
            ]
        });
        if (!encounter) {
            throw new Error(`Encounter not found: ${encounterId}`);
        }
        const patientType = encounter.patientBillingTypeId || encounter?.patient?.additionalData?.[0]?.patientBillingTypeId;
        const patientDOB = encounter?.patient?.dateOfBirth;
        const facilityId = encounter?.location?.facilityId;
        const priceLists = await this.findAll({
            where: {
                visibilityStatus: _constants.VISIBILITY_STATUSES.CURRENT
            },
            order: [
                [
                    'createdAt',
                    'ASC'
                ],
                [
                    'code',
                    'ASC'
                ]
            ]
        });
        const matches = [];
        for (const priceList of priceLists){
            const rules = priceList.rules ?? {};
            const match = (0, _invoicePriceListMatching.equalsIfPresent)(rules.facilityId, facilityId) && (0, _invoicePriceListMatching.equalsIfPresent)(rules.patientType, patientType) && (0, _invoicePriceListMatching.matchesAgeIfPresent)(rules.patientAge, patientDOB);
            if (match) {
                matches.push({
                    id: priceList.id,
                    name: priceList.name || priceList.code
                });
            }
        }
        if (matches.length > 1) {
            throw new Error(`Multiple price lists match the provided inputs: ${matches.map((match)=>match.name).join(', ')}`);
        }
        // Returns null if no matches are found
        return matches[0] ? matches[0].id : null;
    }
};

//# sourceMappingURL=InvoicePriceList.js.map