"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "fhirRoutes", {
    enumerable: true,
    get: function() {
        return fhirRoutes;
    }
});
const _express = require("express");
const _constants = require("@tamanu/constants");
const _fhir = require("@tamanu/shared/utils/fhir");
const _resources = require("@tamanu/shared/utils/fhir/resources");
const _logging = require("@tamanu/shared/services/logging");
const _requireClientHeaders = require("../../middleware/requireClientHeaders");
const _patientSummary = require("./patientSummary");
const _read = require("./read");
const _search = require("./search");
const _create = require("./create");
function fhirRoutes(ctx, { requireClientHeaders } = {}) {
    const routes = (0, _express.Router)();
    routes.use((req, res, next)=>{
        if (![
            'HEAD',
            'GET'
        ].includes(req.method)) {
            const { FhirWriteLog } = req.store.models;
            setImmediate(async ()=>{
                try {
                    await FhirWriteLog.fromRequest(req);
                } catch (err) {
                    _logging.log.error('failed to log FHIR write', {
                        err
                    });
                }
            }).unref();
        }
        res.header('Content-Type', 'application/fhir+json; fhirVersion=4.3');
        next();
    });
    if (requireClientHeaders) {
        routes.use(_requireClientHeaders.requireClientHeaders);
    }
    routes.get(`/Patient/:id/([$])summary`, (0, _patientSummary.patientSummaryHandler)());
    const { models } = ctx.store;
    for (const Resource of (0, _resources.resourcesThatCanDo)(models, _constants.FHIR_INTERACTIONS.INSTANCE.READ)){
        routes.get(`/${Resource.fhirName}/:id`, (0, _read.readHandler)(Resource));
    }
    for (const Resource of (0, _resources.resourcesThatCanDo)(models, _constants.FHIR_INTERACTIONS.TYPE.SEARCH)){
        routes.get(`/${Resource.fhirName}`, (0, _search.searchHandler)(Resource));
    }
    for (const Resource of (0, _resources.resourcesThatCanDo)(models, _constants.FHIR_INTERACTIONS.TYPE.CREATE)){
        routes.post(`/${Resource.fhirName}`, (0, _create.createHandler)(Resource));
    }
    routes.use((err, _req, res, next)=>{
        if (res.headersSent) {
            next(err);
            return;
        }
        const oo = new _fhir.OperationOutcome([
            err
        ]);
        res.status(oo.status()).send(oo.asFhir());
    });
    return routes;
}

//# sourceMappingURL=handlers.js.map