"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "VRSRemote", {
    enumerable: true,
    get: function() {
        return VRSRemote;
    }
});
const _logging = require("@tamanu/shared/services/logging");
const _errors = require("@tamanu/errors");
const _utils = require("@tamanu/shared/utils");
const _VRSPatientAdapter = require("./VRSPatientAdapter");
const _schema = /*#__PURE__*/ _interop_require_wildcard(require("./schema"));
function _define_property(obj, key, value) {
    if (key in obj) {
        Object.defineProperty(obj, key, {
            value: value,
            enumerable: true,
            configurable: true,
            writable: true
        });
    } else {
        obj[key] = value;
    }
    return obj;
}
function _getRequireWildcardCache(nodeInterop) {
    if (typeof WeakMap !== "function") return null;
    var cacheBabelInterop = new WeakMap();
    var cacheNodeInterop = new WeakMap();
    return (_getRequireWildcardCache = function(nodeInterop) {
        return nodeInterop ? cacheNodeInterop : cacheBabelInterop;
    })(nodeInterop);
}
function _interop_require_wildcard(obj, nodeInterop) {
    if (!nodeInterop && obj && obj.__esModule) {
        return obj;
    }
    if (obj === null || typeof obj !== "object" && typeof obj !== "function") {
        return {
            default: obj
        };
    }
    var cache = _getRequireWildcardCache(nodeInterop);
    if (cache && cache.has(obj)) {
        return cache.get(obj);
    }
    var newObj = {
        __proto__: null
    };
    var hasPropertyDescriptor = Object.defineProperty && Object.getOwnPropertyDescriptor;
    for(var key in obj){
        if (key !== "default" && Object.prototype.hasOwnProperty.call(obj, key)) {
            var desc = hasPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : null;
            if (desc && (desc.get || desc.set)) {
                Object.defineProperty(newObj, key, desc);
            } else {
                newObj[key] = obj[key];
            }
        }
    }
    newObj.default = obj;
    if (cache) {
        cache.set(obj, newObj);
    }
    return newObj;
}
const encodeParams = (params)=>Object.entries(params).map((pair)=>pair.map((s)=>encodeURIComponent(s)).join('=')).join('&');
const AUTH_FAILED_STATUS = 401;
let VRSRemote = class VRSRemote {
    clearToken() {
        this.token = null;
        this.tokenExpiry = 0;
    }
    async refreshTokenIfInvalid() {
        // calculate whether we need to fetch a new token
        const remainingTime = this.tokenExpiry - Date.now() - this.tokenExpiryMarginMs;
        const willExpireSoon = remainingTime <= 0;
        const hasToken = !!this.token;
        if (!willExpireSoon && hasToken) {
            return; // no reason to refresh the token
        }
        _logging.log.debug('VRSRemote.refreshTokenIfInvalid(): refreshing token...');
        // make token request
        const body = encodeParams({
            grant_type: 'password',
            username: this.username,
            password: this.password
        });
        const { access_token: accessToken, expires_in: tokenLifetimeSecs } = await this.fetch('/token', {
            validateSchema: _schema.remoteResponse.token,
            shouldRefreshToken: false,
            method: 'POST',
            headers: {},
            body
        });
        const tokenLifetimeMs = tokenLifetimeSecs * 1000;
        // update token info
        this.token = accessToken;
        this.tokenExpiry = Date.now() + tokenLifetimeMs;
        _logging.log.debug(`VRSRemote.refreshTokenIfInvalid(): refreshed token (expires in ${tokenLifetimeMs}ms)`);
    }
    async fetch(path, options = {}) {
        const { shouldRefreshToken = true, validateSchema = null, ...fetchOptions } = options;
        if (!validateSchema) {
            throw new _errors.ValidationError(`VRSRemote.fetch(): must supply a schema to validate against for path ${path}`);
        }
        // refresh token if we think it's expired
        if (shouldRefreshToken) {
            await this.refreshTokenIfInvalid();
        }
        // attempt fetch
        const url = `${this.host}${path}`;
        _logging.log.debug(`VRSRemote.fetch(): fetching ${url}...`);
        const response = await this.fetchImplementation(url, {
            headers: fetchOptions.headers || {
                'Content-Type': 'application/json',
                Accepts: 'application/json',
                Authorization: `Bearer ${this.token}`
            },
            ...fetchOptions
        });
        // handle auth failure by clearing and refreshing token then retrying
        if (shouldRefreshToken && response.status === AUTH_FAILED_STATUS) {
            this.clearToken();
            await this.refreshTokenIfInvalid();
            return this.fetch(path, {
                ...fetchOptions,
                validateSchema,
                shouldRefreshToken: false
            });
        }
        // throw on other errors
        if (!response.ok) {
            const errPayload = JSON.stringify(await (0, _utils.getResponseJsonSafely)(response));
            throw new _errors.RemoteCallError(`VRSRemote.fetch(): Received ${response.status} while calling ${url}`).withExtraData({
                remoteStatus: response.status,
                remoteUrl: url,
                payload: errPayload
            });
        }
        // parse, validate, and return body on success
        const body = await response.json();
        _logging.log.debug(`VRSRemote.fetch(): received body ${JSON.stringify(body)}`);
        const data = await validateSchema.validate(body, {
            stripUnknown: true
        });
        _logging.log.debug(`VRSRemote.fetch(): fetched ${url}`);
        return data;
    }
    async getPatientByFetchId(fetchId) {
        const { data: vrsPatient } = await this.fetch(`/api/Tamanu/Fetch?${encodeParams({
            fetch_id: fetchId
        })}`, {
            validateSchema: _schema.remoteResponse.fetchPatient
        });
        return this.patientAdapter.toTamanu(vrsPatient);
    }
    async acknowledge(fetchId) {
        await this.fetch(`/api/Tamanu/Acknowledge?${encodeParams({
            fetch_id: fetchId
        })}`, {
            validateSchema: _schema.remoteResponse.acknowledge,
            method: 'POST',
            headers: {
                'Content-Length': 0,
                Accepts: 'application/json',
                Authorization: `Bearer ${this.token}`
            }
        });
    }
    async getAllPendingActions() {
        const { data } = await this.fetch('/api/Tamanu/FetchAllPendingActions', {
            validateSchema: _schema.remoteResponse.fetchAllPendingActions
        });
        return data;
    }
    constructor(store, { host, username, password, tokenExpiryMarginMs }){
        _define_property(this, "token", null);
        _define_property(this, "tokenExpiry", 0) // timestamp of token expiry in milliseconds
        ;
        _define_property(this, "fetchImplementation", fetch) // overridden in tests
        ;
        this.patientAdapter = new _VRSPatientAdapter.VRSPatientAdapter(store);
        this.store = store;
        this.host = host;
        this.username = username;
        this.password = password;
        this.tokenExpiryMarginMs = tokenExpiryMarginMs;
    }
};

//# sourceMappingURL=VRSRemote.js.map