"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "locationAssignmentsRouter", {
    enumerable: true,
    get: function() {
        return locationAssignmentsRouter;
    }
});
const _express = /*#__PURE__*/ _interop_require_default(require("express"));
const _expressasynchandler = /*#__PURE__*/ _interop_require_default(require("express-async-handler"));
const _zod = require("zod");
const _sequelize = require("sequelize");
const _datefns = require("date-fns");
const _appointmentScheduling = require("@tamanu/utils/appointmentScheduling");
const _errors = require("@tamanu/errors");
const _dateTime = require("@tamanu/utils/dateTime");
const _constants = require("@tamanu/constants");
function _interop_require_default(obj) {
    return obj && obj.__esModule ? obj : {
        default: obj
    };
}
const locationAssignmentsRouter = _express.default.Router();
const customAssignmentValidation = (data, ctx)=>{
    const repeatingFields = [
        data.repeatEndDate,
        data.repeatFrequency,
        data.repeatUnit
    ];
    const existCount = repeatingFields.filter((v)=>v !== undefined && v !== null).length;
    if (existCount !== repeatingFields.length && existCount !== 0) {
        ctx.addIssue({
            code: _zod.z.ZodIssueCode.custom,
            message: 'Repeat end date, frequency and unit are required for repeating assignments',
            path: [
                'repeatEndDate',
                'repeatFrequency',
                'repeatUnit'
            ]
        });
    }
    if ((0, _datefns.isBefore)((0, _datefns.parseISO)(data.endTime), (0, _datefns.parseISO)(data.startTime))) {
        ctx.addIssue({
            code: _zod.z.ZodIssueCode.custom,
            message: 'Start time must be before end time',
            path: [
                'startTime',
                'endTime'
            ]
        });
    }
};
const getLocationAssignmentsSchema = _zod.z.object({
    after: _dateTime.dateCustomValidation.optional(),
    before: _dateTime.dateCustomValidation.optional(),
    userId: _zod.z.string().optional(),
    locationId: _zod.z.string().optional(),
    facilityId: _zod.z.string().optional(),
    page: _zod.z.coerce.number().int().min(0).optional().default(0),
    rowsPerPage: _zod.z.coerce.number().int().min(1).optional().default(50),
    all: _zod.z.string().optional().default('false').transform((value)=>value.toLowerCase() === 'true')
});
locationAssignmentsRouter.get('/', (0, _expressasynchandler.default)(async (req, res)=>{
    req.checkPermission('list', 'LocationSchedule');
    const { LocationAssignment, LocationAssignmentTemplate, User, Location, LocationGroup } = req.models;
    const query = await getLocationAssignmentsSchema.parseAsync(req.query);
    const { after, before, locationId, facilityId, userId, page, rowsPerPage, all } = query;
    const includeOptions = [
        {
            model: User,
            as: 'user',
            attributes: [
                'id',
                'displayName',
                'email'
            ]
        },
        {
            model: LocationAssignmentTemplate,
            as: 'template',
            attributes: [
                'id',
                'date',
                'startTime',
                'endTime',
                'repeatFrequency',
                'repeatUnit',
                'repeatEndDate'
            ]
        },
        {
            model: Location,
            as: 'location',
            attributes: [
                'id',
                'name',
                'facilityId'
            ],
            include: [
                {
                    model: LocationGroup,
                    as: 'locationGroup',
                    attributes: [
                        'id',
                        'name',
                        'facilityId'
                    ]
                }
            ]
        }
    ];
    const filters = {
        date: {
            ...after && {
                [_sequelize.Op.gte]: after
            },
            ...before && {
                [_sequelize.Op.lte]: before
            }
        },
        ...locationId && {
            locationId
        },
        ...userId && {
            userId
        },
        ...facilityId && {
            [_sequelize.Op.or]: [
                {
                    '$location.facility_id$': facilityId
                },
                {
                    '$location.locationGroup.facility_id$': facilityId
                }
            ]
        }
    };
    const { rows, count } = await LocationAssignment.findAndCountAll({
        include: includeOptions,
        where: filters,
        limit: all ? undefined : rowsPerPage,
        offset: all ? undefined : page * rowsPerPage,
        order: [
            [
                'date',
                'ASC'
            ],
            [
                'startTime',
                'ASC'
            ]
        ]
    });
    res.send({
        count,
        data: rows
    });
}));
const createLocationAssignmentSchema = _zod.z.object({
    userId: _zod.z.string(),
    locationId: _zod.z.string(),
    date: _dateTime.dateCustomValidation,
    startTime: _dateTime.timeCustomValidation,
    endTime: _dateTime.timeCustomValidation,
    repeatEndDate: _dateTime.dateCustomValidation.nullable().optional(),
    repeatFrequency: _zod.z.number().int().positive().optional(),
    repeatUnit: _zod.z.enum(_constants.REPEAT_FREQUENCY_VALUES).optional()
}).superRefine(customAssignmentValidation);
locationAssignmentsRouter.post('/', (0, _expressasynchandler.default)(async (req, res)=>{
    req.checkPermission('create', 'LocationSchedule');
    const body = await createLocationAssignmentSchema.parseAsync(req.body);
    const { User } = req.models;
    const clinician = await User.findByPk(body.userId);
    if (!clinician) {
        throw new _errors.NotFoundError(`User not found`);
    }
    const maxFutureMonths = await req.settings.get('locationAssignments.assignmentMaxFutureMonths');
    const maxAssignmentDate = (0, _datefns.addMonths)(new Date(), maxFutureMonths);
    if ((0, _datefns.isAfter)((0, _datefns.parseISO)(body.date), maxAssignmentDate)) {
        throw new _errors.InvalidOperationError(`Date should not be greater than ${(0, _dateTime.toDateString)(maxAssignmentDate)}`);
    }
    if (body.repeatEndDate && (0, _datefns.isAfter)((0, _datefns.parseISO)(body.repeatEndDate), maxAssignmentDate)) {
        throw new _errors.InvalidOperationError(`End date should not be greater than ${(0, _dateTime.toDateString)(maxAssignmentDate)}`);
    }
    const overlapAssignments = await findOverlappingAssignments(req.models, body);
    if (overlapAssignments?.length > 0) {
        res.status(400).send({
            error: {
                message: 'Location assignment overlaps with existing assignments',
                type: 'overlap_assignment_error',
                overlapAssignments
            }
        });
        return;
    }
    if (body.repeatFrequency) {
        await createRepeatingLocationAssignment(req, body);
    } else {
        await createSingleLocationAssignment(req, body);
    }
    res.status(201).send({
        success: true
    });
}));
const updateLocationAssignmentSchema = _zod.z.object({
    userId: _zod.z.string(),
    locationId: _zod.z.string(),
    date: _dateTime.dateCustomValidation,
    startTime: _dateTime.timeCustomValidation,
    endTime: _dateTime.timeCustomValidation,
    repeatEndDate: _zod.z.string().nullable().optional(),
    repeatFrequency: _zod.z.number().int().positive().optional(),
    repeatUnit: _zod.z.enum(_constants.REPEAT_FREQUENCY_VALUES).optional(),
    updateAllNextRecords: _zod.z.boolean().default(false).optional()
}).superRefine(customAssignmentValidation);
locationAssignmentsRouter.put('/:id', (0, _expressasynchandler.default)(async (req, res)=>{
    req.checkPermission('write', 'LocationSchedule');
    const { id } = req.params;
    const body = await updateLocationAssignmentSchema.parseAsync(req.body);
    const { LocationAssignment } = req.models;
    const assignment = await LocationAssignment.findByPk(id);
    if (!assignment) {
        throw new _errors.InvalidOperationError('Location assignment not found');
    }
    const maxFutureMonths = await req.settings.get('locationAssignments.assignmentMaxFutureMonths');
    const maxAssignmentDate = (0, _datefns.addMonths)(new Date(), maxFutureMonths);
    if ((0, _datefns.isAfter)((0, _datefns.parseISO)(body.date), maxAssignmentDate)) {
        throw new _errors.InvalidOperationError(`Date should not be greater than ${(0, _dateTime.toDateString)(maxAssignmentDate)}`);
    }
    if (body.repeatEndDate && (0, _datefns.isAfter)((0, _datefns.parseISO)(body.repeatEndDate), maxAssignmentDate)) {
        throw new _errors.InvalidOperationError(`End date should not be greater than ${(0, _dateTime.toDateString)(maxAssignmentDate)}`);
    }
    let result;
    // If the assignment is not repeating
    if (!assignment.templateId) {
        result = await updateNonRepeatingAssignment(req, body, assignment);
    } else {
        if (body.updateAllNextRecords) {
            const maxFutureMonths = await req.settings.get('locationAssignments.assignmentMaxFutureMonths');
            if ((0, _datefns.differenceInMonths)((0, _datefns.parseISO)(body.repeatEndDate), new Date()) > maxFutureMonths) {
                throw new _errors.InvalidOperationError(`End date should be within ${maxFutureMonths} months from today`);
            }
            if ((0, _datefns.differenceInMonths)((0, _datefns.parseISO)(body.date), new Date()) > maxFutureMonths) {
                throw new _errors.InvalidOperationError(`Date should be within ${maxFutureMonths} months from today`);
            }
            result = await updateFutureAssignments(req, body, assignment);
        } else {
            result = await updateSingleRepeatingAssignment(req, body, assignment);
        }
    }
    if (result.overlapAssignments?.length > 0) {
        res.status(400).send({
            error: {
                message: 'Location assignment overlaps with existing assignments',
                type: 'overlap_assignment_error',
                overlapAssignments: result.overlapAssignments
            }
        });
        return;
    }
    res.status(200).send({
        success: true
    });
}));
const deleteLocationAssignmentSchema = _zod.z.object({
    deleteAllNextRecords: _zod.z.string().optional().default('false').transform((value)=>value.toLowerCase() === 'true')
});
locationAssignmentsRouter.delete('/:id', (0, _expressasynchandler.default)(async (req, res)=>{
    req.checkPermission('delete', 'LocationSchedule');
    const { id } = req.params;
    const { models, db } = req;
    const { LocationAssignment } = models;
    const query = await deleteLocationAssignmentSchema.parseAsync(req.query);
    const assignment = await LocationAssignment.findByPk(id);
    if (!assignment) {
        throw new _errors.InvalidOperationError('Location assignment not found');
    }
    if (query.deleteAllNextRecords && !assignment.templateId) {
        throw new _errors.InvalidOperationError('Cannot delete future assignments for non-repeating assignments');
    }
    if (query.deleteAllNextRecords) {
        await db.transaction(async ()=>{
            await deleteSelectedAndFutureAssignments(models, assignment.templateId, assignment.date);
        });
    } else {
        await assignment.destroy();
    }
    res.status(200).send({
        success: true
    });
}));
const overlappingAssignmentsSchema = _zod.z.object({
    id: _zod.z.string().optional(),
    userId: _zod.z.string(),
    locationId: _zod.z.string(),
    date: _dateTime.dateCustomValidation,
    startTime: _dateTime.timeCustomValidation,
    endTime: _dateTime.timeCustomValidation,
    repeatEndDate: _dateTime.dateCustomValidation.nullable().optional(),
    repeatFrequency: _zod.z.number().int().positive().optional(),
    repeatUnit: _zod.z.enum(_constants.REPEAT_FREQUENCY_VALUES).optional()
}).superRefine(customAssignmentValidation);
locationAssignmentsRouter.post('/overlapping-assignments', (0, _expressasynchandler.default)(async (req, res)=>{
    const { models } = req;
    req.checkPermission('list', 'LocationSchedule');
    const body = await overlappingAssignmentsSchema.parseAsync(req.body);
    const maxFutureMonths = await req.settings.get('locationAssignments.assignmentMaxFutureMonths');
    const maxAssignmentDate = (0, _datefns.addMonths)(new Date(), maxFutureMonths);
    if ((0, _datefns.isAfter)((0, _datefns.parseISO)(body.date), maxAssignmentDate)) {
        throw new _errors.InvalidOperationError(`Date should not be greater than ${(0, _dateTime.toDateString)(maxAssignmentDate)}`);
    }
    if (body.repeatEndDate && (0, _datefns.isAfter)((0, _datefns.parseISO)(body.repeatEndDate), maxAssignmentDate)) {
        throw new _errors.InvalidOperationError(`End date should not be greater than ${(0, _dateTime.toDateString)(maxAssignmentDate)}`);
    }
    const excludeAssignmentIds = [];
    if (body.id) {
        const assignment = await models.LocationAssignment.findByPk(body.id);
        const template = await models.LocationAssignmentTemplate.findByPk(assignment.templateId);
        const excludeAssignments = await models.LocationAssignment.findAll({
            where: {
                templateId: template.id,
                date: {
                    [_sequelize.Op.gte]: assignment.date
                }
            }
        });
        excludeAssignmentIds.push(...excludeAssignments.map((assignment)=>assignment.id));
    }
    const overlapAssignments = await findOverlappingAssignments(req.models, body, {
        excludeAssignmentIds
    });
    res.send(overlapAssignments);
}));
const overlappingLeavesSchema = _zod.z.object({
    userId: _zod.z.string(),
    date: _dateTime.dateCustomValidation,
    repeatEndDate: _dateTime.dateCustomValidation.nullable().optional(),
    repeatFrequency: _zod.z.coerce.number().int().positive().optional(),
    repeatUnit: _zod.z.enum(_constants.REPEAT_FREQUENCY_VALUES).optional()
}).superRefine(customAssignmentValidation);
locationAssignmentsRouter.get('/overlapping-leaves', (0, _expressasynchandler.default)(async (req, res)=>{
    req.checkPermission('read', 'User');
    const { UserLeave } = req.models;
    const query = await overlappingLeavesSchema.parseAsync(req.query);
    let assignmentDates = [
        query.date
    ];
    if (query.repeatFrequency) {
        assignmentDates = (0, _appointmentScheduling.generateFrequencyDates)(query.date, query.repeatEndDate, query.repeatFrequency, query.repeatUnit);
    }
    const userLeaves = await UserLeave.findAll({
        where: {
            userId: query.userId,
            endDate: {
                [_sequelize.Op.gte]: assignmentDates[0]
            },
            startDate: {
                [_sequelize.Op.lte]: assignmentDates.at(-1)
            }
        },
        attributes: [
            'id',
            'startDate',
            'endDate',
            'userId'
        ],
        order: [
            [
                'startDate',
                'ASC'
            ]
        ]
    });
    const overlappingLeaves = userLeaves.filter((leave)=>{
        return assignmentDates.some((date)=>leave.startDate <= date && date <= leave.endDate);
    });
    res.send(overlappingLeaves);
}));
/**
 * Create a repeating location assignment with template and generate initial assignment
 */ async function createRepeatingLocationAssignment(req, body) {
    const { models: { LocationAssignmentTemplate }, db } = req;
    const { userId, locationId, startTime, endTime, date, repeatEndDate, repeatUnit, repeatFrequency } = body;
    await db.transaction(async ()=>{
        const template = await LocationAssignmentTemplate.create({
            userId,
            locationId,
            startTime,
            endTime,
            date,
            repeatEndDate,
            repeatFrequency,
            repeatUnit
        });
        await template.generateRepeatingLocationAssignments();
    });
}
async function createSingleLocationAssignment(req, body) {
    const { LocationAssignment } = req.models;
    const { userId, locationId, date, startTime, endTime } = body;
    await checkUserLeaveStatus(req.models, userId, date);
    await LocationAssignment.create({
        userId,
        locationId,
        date,
        startTime,
        endTime
    });
}
async function updateNonRepeatingAssignment(req, body, assignment) {
    await checkUserLeaveStatus(req.models, assignment.userId, body.date);
    const overlapAssignments = await findOverlappingAssignments(req.models, {
        locationId: body.locationId,
        date: body.date,
        startTime: body.startTime,
        endTime: body.endTime
    }, {
        excludeAssignmentIds: [
            assignment.id
        ]
    });
    if (overlapAssignments?.length > 0) {
        return {
            success: false,
            overlapAssignments
        };
    }
    await assignment.update({
        locationId: body.locationId,
        date: body.date,
        startTime: body.startTime,
        endTime: body.endTime
    });
    return {
        success: true
    };
}
async function updateSingleRepeatingAssignment(req, body, assignment) {
    const { models, db } = req;
    const { LocationAssignment } = models;
    let overlapAssignments = [];
    try {
        await db.transaction(async ()=>{
            await assignment.destroy();
            await checkUserLeaveStatus(req.models, assignment.userId, body.date);
            overlapAssignments = await findOverlappingAssignments(req.models, {
                locationId: body.locationId,
                date: body.date,
                startTime: body.startTime,
                endTime: body.endTime
            });
            if (overlapAssignments?.length > 0) {
                throw new _errors.InvalidOperationError('Location assignment overlaps with existing assignments');
            }
            await LocationAssignment.create({
                userId: assignment.userId,
                locationId: body.locationId,
                date: body.date,
                startTime: body.startTime,
                endTime: body.endTime
            });
        });
        return {
            success: true
        };
    } catch (error) {
        if (overlapAssignments?.length > 0) {
            return {
                success: false,
                overlapAssignments
            };
        }
        throw error;
    }
}
async function updateFutureAssignments(req, body, assignment) {
    const { models, db } = req;
    const { LocationAssignmentTemplate } = models;
    let overlapAssignments = [];
    try {
        await db.transaction(async ()=>{
            const template = await LocationAssignmentTemplate.findByPk(assignment.templateId);
            await deleteSelectedAndFutureAssignments(models, template.id, assignment.date);
            overlapAssignments = await findOverlappingAssignments(models, body);
            if (overlapAssignments?.length > 0) {
                throw new _errors.InvalidOperationError('Location assignment overlaps with existing assignments');
            }
            const newTemplate = await LocationAssignmentTemplate.create({
                userId: template.userId,
                locationId: body.locationId,
                startTime: body.startTime,
                endTime: body.endTime,
                date: body.date,
                repeatEndDate: body.repeatEndDate,
                repeatFrequency: body.repeatFrequency,
                repeatUnit: body.repeatUnit
            });
            await newTemplate.generateRepeatingLocationAssignments();
        });
        return {
            success: true
        };
    } catch (error) {
        if (overlapAssignments?.length > 0) {
            return {
                success: false,
                overlapAssignments
            };
        }
        throw error;
    }
}
async function deleteSelectedAndFutureAssignments(models, templateId, assignmentDate) {
    const { LocationAssignment, LocationAssignmentTemplate } = models;
    // Delete selected and future assignments for repeating location assignments
    await LocationAssignment.destroy({
        where: {
            templateId,
            date: {
                [_sequelize.Op.gte]: assignmentDate
            }
        }
    });
    // Get the latest non-deleted assignment
    const latestAssignment = await LocationAssignment.findOne({
        where: {
            templateId
        },
        order: [
            [
                'date',
                'DESC'
            ]
        ]
    });
    if (!latestAssignment) {
        return await LocationAssignmentTemplate.destroy({
            where: {
                id: templateId
            }
        });
    }
    // Update the repeat end date to the latest assignment date
    await LocationAssignmentTemplate.update({
        repeatEndDate: latestAssignment.date
    }, {
        where: {
            id: templateId
        }
    });
}
/**
 * Check if the new assignment overlaps with existing generated assignments.
 */ async function findOverlappingAssignments(models, body, options = {}) {
    const { LocationAssignment, User } = models;
    const { locationId, date, startTime, endTime, repeatFrequency, repeatUnit, repeatEndDate, userId } = body;
    let dateFilter = {
        [_sequelize.Op.eq]: date
    };
    if (repeatFrequency) {
        const assignmentDates = (0, _appointmentScheduling.generateFrequencyDates)(date, repeatEndDate, repeatFrequency, repeatUnit);
        dateFilter = {
            [_sequelize.Op.in]: assignmentDates
        };
    }
    const overlappingAssignments = await LocationAssignment.findAll({
        include: [
            {
                model: User,
                as: 'user',
                attributes: [
                    'id',
                    'displayName',
                    'email'
                ]
            }
        ],
        where: {
            locationId,
            startTime: {
                [_sequelize.Op.lt]: endTime
            },
            endTime: {
                [_sequelize.Op.gt]: startTime
            },
            date: dateFilter,
            ...options.excludeAssignmentIds && {
                id: {
                    [_sequelize.Op.notIn]: options.excludeAssignmentIds
                }
            },
            ...repeatFrequency && {
                [_sequelize.Op.or]: [
                    // All records with null templateId
                    {
                        templateId: null
                    },
                    // First record for each templateId
                    {
                        templateId: {
                            [_sequelize.Op.ne]: null
                        },
                        id: {
                            [_sequelize.Op.in]: (0, _sequelize.literal)(`(
                SELECT DISTINCT ON ("template_id") id
                FROM "location_assignments"
                WHERE template_id IS NOT NULL
                ${options.excludeAssignmentIds?.length > 0 ? 'AND id NOT IN (:excludeAssignmentIds)' : ''}
                AND date IN (:dates)
                AND NOT EXISTS (SELECT 1 FROM user_leaves WHERE user_id = :userId AND date BETWEEN start_date AND end_date)
                AND location_id = :locationId
                AND deleted_at IS NULL
                ORDER BY template_id, date ASC
              )`)
                        }
                    }
                ]
            }
        },
        replacements: {
            dates: dateFilter[_sequelize.Op.in],
            locationId,
            userId,
            ...options.excludeAssignmentIds?.length > 0 && {
                excludeAssignmentIds: options.excludeAssignmentIds
            }
        },
        attributes: [
            'id',
            'locationId',
            'date',
            'startTime',
            'endTime',
            'templateId'
        ],
        order: [
            [
                'date',
                'ASC'
            ],
            [
                'startTime',
                'ASC'
            ]
        ]
    });
    return overlappingAssignments.map((assignment)=>({
            id: assignment.id,
            date: assignment.date,
            startTime: assignment.startTime,
            endTime: assignment.endTime,
            locationId: assignment.locationId,
            user: assignment.user,
            templateId: assignment.templateId,
            isRepeating: !!assignment.templateId
        }));
}
async function checkUserLeaveStatus(models, userId, date) {
    const { UserLeave } = models;
    const userLeave = await UserLeave.findOne({
        where: {
            userId,
            startDate: {
                [_sequelize.Op.lte]: date
            },
            endDate: {
                [_sequelize.Op.gte]: date
            }
        },
        attributes: [
            'id'
        ]
    });
    if (userLeave) {
        throw new _errors.InvalidOperationError(`User is on leave!`);
    }
}

//# sourceMappingURL=locationAssignments.js.map