"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "refresh", {
    enumerable: true,
    get: function() {
        return refresh;
    }
});
const _config = /*#__PURE__*/ _interop_require_default(require("config"));
const _expressasynchandler = /*#__PURE__*/ _interop_require_default(require("express-async-handler"));
const _bcrypt = /*#__PURE__*/ _interop_require_default(require("bcrypt"));
const _jose = /*#__PURE__*/ _interop_require_wildcard(require("jose"));
const _constants = require("@tamanu/constants");
const _errors = require("@tamanu/errors");
const _utils = require("./utils");
function _interop_require_default(obj) {
    return obj && obj.__esModule ? obj : {
        default: obj
    };
}
function _getRequireWildcardCache(nodeInterop) {
    if (typeof WeakMap !== "function") return null;
    var cacheBabelInterop = new WeakMap();
    var cacheNodeInterop = new WeakMap();
    return (_getRequireWildcardCache = function(nodeInterop) {
        return nodeInterop ? cacheNodeInterop : cacheBabelInterop;
    })(nodeInterop);
}
function _interop_require_wildcard(obj, nodeInterop) {
    if (!nodeInterop && obj && obj.__esModule) {
        return obj;
    }
    if (obj === null || typeof obj !== "object" && typeof obj !== "function") {
        return {
            default: obj
        };
    }
    var cache = _getRequireWildcardCache(nodeInterop);
    if (cache && cache.has(obj)) {
        return cache.get(obj);
    }
    var newObj = {
        __proto__: null
    };
    var hasPropertyDescriptor = Object.defineProperty && Object.getOwnPropertyDescriptor;
    for(var key in obj){
        if (key !== "default" && Object.prototype.hasOwnProperty.call(obj, key)) {
            var desc = hasPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : null;
            if (desc && (desc.get || desc.set)) {
                Object.defineProperty(newObj, key, desc);
            } else {
                newObj[key] = obj[key];
            }
        }
    }
    newObj.default = obj;
    if (cache) {
        cache.set(obj, newObj);
    }
    return newObj;
}
const refresh = (0, _expressasynchandler.default)(async (req, res)=>{
    const { body, store } = req;
    const { refreshToken, deviceId } = body;
    const { auth: { saltRounds, secret, tokenDuration, refreshToken: { refreshIdLength, tokenDuration: refreshTokenDuration, absoluteExpiration, secret: refreshSecret } }, canonicalHostName } = _config.default;
    if (!(0, _utils.isInternalClient)(req.header('X-Tamanu-Client'))) {
        throw new _errors.InvalidCredentialError('Refresh tokens are only available to internal clients');
    }
    let contents = null;
    try {
        contents = await (0, _utils.verifyToken)(refreshToken, refreshSecret, {
            audience: _constants.JWT_TOKEN_TYPES.REFRESH,
            issuer: canonicalHostName
        });
    } catch (e) {
        throw new _errors.InvalidTokenError('validity');
    }
    const { userId, refreshId } = contents.payload;
    const user = await store.models.User.findOne({
        where: {
            id: userId,
            visibilityStatus: _constants.VISIBILITY_STATUSES.CURRENT
        }
    });
    if (!user) {
        throw new _errors.InvalidTokenError('user');
    }
    const dbEntry = await store.models.RefreshToken.findOne({
        where: {
            userId,
            deviceId
        }
    });
    if (!dbEntry) {
        throw new _errors.InvalidTokenError('row');
    }
    if (dbEntry.expiresAt < new Date()) {
        throw new _errors.InvalidTokenError('expired');
    }
    const refreshIdValid = await _bcrypt.default.compare(refreshId, dbEntry.refreshId);
    if (!refreshIdValid) {
        throw new _errors.InvalidTokenError('id');
    }
    // issue new access token
    const accessTokenJwtId = (0, _utils.getRandomU32)();
    const token = await (0, _utils.buildToken)({
        userId: user.id,
        deviceId
    }, secret, {
        expiresIn: tokenDuration,
        audience: _constants.JWT_TOKEN_TYPES.ACCESS,
        issuer: canonicalHostName,
        jwtid: `${accessTokenJwtId}`
    });
    // rotate refresh token
    const newRefreshId = await (0, _utils.getRandomBase64String)(refreshIdLength);
    const refreshTokenJwtId = (0, _utils.getRandomU32)();
    const hashedRefreshId = await _bcrypt.default.hash(newRefreshId, saltRounds);
    const newRefreshToken = await (0, _utils.buildToken)({
        userId: user.id,
        refreshId: newRefreshId,
        // If absolute expiration pass through the exp from the old token
        ...absoluteExpiration && {
            exp: contents.exp
        }
    }, refreshSecret, {
        audience: _constants.JWT_TOKEN_TYPES.REFRESH,
        issuer: canonicalHostName,
        jwtid: `${refreshTokenJwtId}`,
        ...!absoluteExpiration && {
            expiresIn: refreshTokenDuration
        }
    });
    // Extract expiry as set by jose.SignJWT
    const { exp } = _jose.decodeJwt(newRefreshToken);
    await store.models.RefreshToken.upsert({
        refreshId: hashedRefreshId,
        userId: user.id,
        deviceId,
        expiresAt: absoluteExpiration ? dbEntry.expiresAt : new Date(exp * 1000)
    }, {
        where: {
            userId: user.id,
            deviceId
        }
    });
    res.send({
        token,
        refreshToken: newRefreshToken
    });
});

//# sourceMappingURL=refresh.js.map