"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
function _export(target, all) {
    for(var name in all)Object.defineProperty(target, name, {
        enumerable: true,
        get: all[name]
    });
}
_export(exports, {
    generateWhereClause: function() {
        return generateWhereClause;
    },
    singleMatch: function() {
        return singleMatch;
    }
});
const _lodash = require("lodash");
const _sequelize = require("sequelize");
const _yup = /*#__PURE__*/ _interop_require_wildcard(require("yup"));
const _constants = require("@tamanu/constants");
const _fhir = require("@tamanu/shared/utils/fhir");
const _common = require("./common");
const _jsonb = require("./jsonb");
function _getRequireWildcardCache(nodeInterop) {
    if (typeof WeakMap !== "function") return null;
    var cacheBabelInterop = new WeakMap();
    var cacheNodeInterop = new WeakMap();
    return (_getRequireWildcardCache = function(nodeInterop) {
        return nodeInterop ? cacheNodeInterop : cacheBabelInterop;
    })(nodeInterop);
}
function _interop_require_wildcard(obj, nodeInterop) {
    if (!nodeInterop && obj && obj.__esModule) {
        return obj;
    }
    if (obj === null || typeof obj !== "object" && typeof obj !== "function") {
        return {
            default: obj
        };
    }
    var cache = _getRequireWildcardCache(nodeInterop);
    if (cache && cache.has(obj)) {
        return cache.get(obj);
    }
    var newObj = {
        __proto__: null
    };
    var hasPropertyDescriptor = Object.defineProperty && Object.getOwnPropertyDescriptor;
    for(var key in obj){
        if (key !== "default" && Object.prototype.hasOwnProperty.call(obj, key)) {
            var desc = hasPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : null;
            if (desc && (desc.get || desc.set)) {
                Object.defineProperty(newObj, key, desc);
            } else {
                newObj[key] = obj[key];
            }
        }
    }
    newObj.default = obj;
    if (cache) {
        cache.set(obj, newObj);
    }
    return newObj;
}
function generateWhereClause(query, parameters, FhirResource) {
    const andWhere = [];
    for (const [name, paramQueries] of query.entries()){
        if (_fhir.RESULT_PARAMETER_NAMES.includes(name)) continue;
        const def = parameters.get(name);
        if (def.path.length === 0) continue;
        for (const paramQuery of paramQueries){
            const alternates = def.path.flatMap(([field, ...path])=>{
                const resolvedPath = [
                    (0, _common.findField)(FhirResource, field).field,
                    ...path
                ];
                return singleMatch(resolvedPath, paramQuery, def, FhirResource);
            });
            andWhere.push({
                [_sequelize.Op.or]: alternates
            });
        }
    }
    // Filter to ensure all returned resources have been fully resolved
    return {
        resolved: true,
        [_sequelize.Op.and]: andWhere
    };
}
const INVERSE_OPS = new Map([
    [
        _sequelize.Op.regexp,
        'OPERATOR(fhir.<~)'
    ],
    [
        _sequelize.Op.iRegexp,
        'OPERATOR(fhir.<~*)'
    ],
    [
        _sequelize.Op.notRegexp,
        'OPERATOR(fhir.<!~)'
    ],
    [
        _sequelize.Op.notIRegexp,
        'OPERATOR(fhir.<!~*)'
    ],
    [
        'OPERATOR(fhir.<~)',
        _sequelize.Op.regexp
    ],
    [
        'OPERATOR(fhir.<~*)',
        _sequelize.Op.iRegexp
    ],
    [
        'OPERATOR(fhir.<!~)',
        _sequelize.Op.notRegexp
    ],
    [
        'OPERATOR(fhir.<!~*)',
        _sequelize.Op.notIRegexp
    ],
    [
        _sequelize.Op.gt,
        _sequelize.Op.lte
    ],
    [
        _sequelize.Op.gte,
        _sequelize.Op.lt
    ],
    [
        _sequelize.Op.lt,
        _sequelize.Op.gte
    ],
    [
        _sequelize.Op.lte,
        _sequelize.Op.gt
    ]
]);
function singleMatch(path, paramQuery, paramDef, Model) {
    return paramQuery.value.map((value)=>{
        const matches = typedMatch(value, paramQuery, paramDef).map(({ op, val, extraPath = [] })=>{
            const entirePath = [
                ...path,
                ...extraPath
            ];
            // optimisation in the simple case
            if (entirePath.length === 1) {
                return _sequelize.Sequelize.where(_sequelize.Sequelize.col(entirePath[0]), op, val);
            }
            const escaped = paramDef.type === _constants.FHIR_SEARCH_PARAMETERS.NUMBER ? val.toString() : Model.sequelize.escape(val);
            // the JSONB queries below are quite complex, and postgres' query planner
            // can't figure out how to optimise them. so we help it out by adding a
            // boolean condition that will let it use a GIN index as a pre-scan filter
            const optimisingCondition = `"${entirePath[0]}" @? '${(0, _jsonb.getJsonbPath)(entirePath)}'`;
            // need to inverse the ops because we're writing the sql in the opposite
            // direction (match operator any(...)) instead of (value operator match)
            const inverseOp = INVERSE_OPS.get(op) ?? op;
            if (typeof inverseOp === 'string') {
                // our custom inverse regex operators don't work with sequelize, so we
                // need to write literals for them. also see:
                // https://github.com/sequelize/sequelize/issues/13011
                // we're just writing the literal
                // instead of being able to use sequelize's utilities.
                // path: ['a', 'b', '[]', 'c', '[]', 'd']
                // sql: value operator ANY(SELECT jsonb_path_query(a, '$.b[*].c[*].d') #>> '{}');
                const selector = `ANY(SELECT jsonb_path_query(${entirePath[0]}, '${(0, _jsonb.getJsonbPath)(entirePath)}') #>> '{}')`;
                return _sequelize.Sequelize.literal(`${escaped} ${inverseOp} ${selector} AND ${optimisingCondition}`);
            }
            // while #>> works regardless of the jsonb path, using
            // explicit function names needs different treatment.
            const selector = _sequelize.Sequelize.fn('any', _sequelize.Sequelize.fn('select', (0, _jsonb.getJsonbQueryFn)(entirePath)));
            return _sequelize.Sequelize.and([
                // actual comparison
                _sequelize.Sequelize.where(_sequelize.Sequelize.literal(escaped), inverseOp, selector),
                _sequelize.Sequelize.literal(optimisingCondition)
            ]);
        });
        return matches.length === 1 ? matches[0] : _sequelize.Sequelize.and(matches);
    });
}
function typedMatch(value, query, def) {
    switch(def.type){
        case _constants.FHIR_SEARCH_PARAMETERS.NUMBER:
            {
                return [
                    {
                        op: prefixToOp(value.prefix),
                        val: value.number
                    }
                ];
            }
        case _constants.FHIR_SEARCH_PARAMETERS.DATE:
            {
                switch(def.datePrecision){
                    case _constants.FHIR_DATETIME_PRECISION.DAYS:
                        return [
                            {
                                op: prefixToOp(value.prefix),
                                val: value.date.sql.split(' ')[0]
                            }
                        ];
                    case _constants.FHIR_DATETIME_PRECISION.SECONDS:
                        return [
                            {
                                op: prefixToOp(value.prefix),
                                val: value.date.sql
                            }
                        ];
                    default:
                        throw new _fhir.Unsupported(`unsupported date precision: ${def.datePrecision}`);
                }
            }
        case _constants.FHIR_SEARCH_PARAMETERS.STRING:
            {
                switch(query.modifier){
                    case undefined:
                    case null:
                    case 'starts-with':
                        return [
                            {
                                op: _sequelize.Op.iRegexp,
                                val: `^${(0, _lodash.escapeRegExp)(value)}.*`
                            }
                        ];
                    case 'ends-with':
                        return [
                            {
                                op: _sequelize.Op.iRegexp,
                                val: `.*${(0, _lodash.escapeRegExp)(value)}$`
                            }
                        ];
                    case 'contains':
                        return [
                            {
                                op: _sequelize.Op.iRegexp,
                                val: `.*${(0, _lodash.escapeRegExp)(value)}.*`
                            }
                        ];
                    case 'exact':
                        return [
                            {
                                op: _sequelize.Op.eq,
                                val: value
                            }
                        ];
                    default:
                        throw new _fhir.Unsupported(`unsupported string modifier: ${query.modifier}`);
                }
            }
        case _constants.FHIR_SEARCH_PARAMETERS.TOKEN:
            {
                const { system, code } = value;
                switch(def.tokenType){
                    case _constants.FHIR_SEARCH_TOKEN_TYPES.CODING:
                    case _constants.FHIR_SEARCH_TOKEN_TYPES.VALUE:
                        {
                            const valuePath = def.tokenType === _constants.FHIR_SEARCH_TOKEN_TYPES.VALUE ? 'value' : 'code';
                            if (system && code) {
                                return [
                                    {
                                        op: _sequelize.Op.eq,
                                        val: system,
                                        extraPath: [
                                            'system'
                                        ]
                                    },
                                    {
                                        op: _sequelize.Op.eq,
                                        val: code,
                                        extraPath: [
                                            valuePath
                                        ]
                                    }
                                ];
                            }
                            if (system) {
                                return [
                                    {
                                        op: _sequelize.Op.eq,
                                        val: system,
                                        extraPath: [
                                            'system'
                                        ]
                                    }
                                ];
                            }
                            if (code) {
                                return [
                                    {
                                        op: _sequelize.Op.eq,
                                        val: code,
                                        extraPath: [
                                            valuePath
                                        ]
                                    }
                                ];
                            }
                            throw new _fhir.Invalid('token searches require either or both of system|code');
                        }
                    case _constants.FHIR_SEARCH_TOKEN_TYPES.BOOLEAN:
                        {
                            return [
                                {
                                    op: _sequelize.Op.eq,
                                    val: _yup.boolean().validateSync(code)
                                }
                            ];
                        }
                    case _constants.FHIR_SEARCH_TOKEN_TYPES.PRESENCE:
                        {
                            const present = _yup.boolean().validateSync(code);
                            return [
                                {
                                    op: present ? _sequelize.Op.not : _sequelize.Op.is,
                                    val: null
                                }
                            ];
                        }
                    case _constants.FHIR_SEARCH_TOKEN_TYPES.STRING:
                        {
                            return [
                                {
                                    op: _sequelize.Op.eq,
                                    val: code
                                }
                            ];
                        }
                    default:
                        throw new _fhir.Unsupported(`unsupported search token type ${def.tokenType}`);
                }
            }
        default:
            throw new _fhir.Unsupported(`unsupported search type ${def.type}`);
    }
}
function prefixToOp(prefix) {
    switch(prefix){
        case null:
        case _constants.FHIR_SEARCH_PREFIXES.EQ:
            return _sequelize.Op.eq;
        case _constants.FHIR_SEARCH_PREFIXES.NE:
            return _sequelize.Op.ne;
        case _constants.FHIR_SEARCH_PREFIXES.LT:
            return _sequelize.Op.lt;
        case _constants.FHIR_SEARCH_PREFIXES.GT:
            return _sequelize.Op.gt;
        case _constants.FHIR_SEARCH_PREFIXES.LE:
            return _sequelize.Op.lte;
        case _constants.FHIR_SEARCH_PREFIXES.GE:
            return _sequelize.Op.gte;
        default:
            throw new _fhir.Unsupported(`unsupported search prefix: ${prefix}`);
    }
}

//# sourceMappingURL=where.js.map