"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "FhirMissingResources", {
    enumerable: true,
    get: function() {
        return FhirMissingResources;
    }
});
const _config = /*#__PURE__*/ _interop_require_default(require("config"));
const _constants = require("@tamanu/constants");
const _tasks = require("@tamanu/shared/tasks");
const _logging = require("@tamanu/shared/services/logging");
const _resources = require("@tamanu/shared/utils/fhir/resources");
const _prepareQuery = require("../utils/prepareQuery");
const _sequelize = require("sequelize");
function _interop_require_default(obj) {
    return obj && obj.__esModule ? obj : {
        default: obj
    };
}
let FhirMissingResources = class FhirMissingResources extends _tasks.ScheduledTask {
    getName() {
        return 'FhirMissingResources';
    }
    async getQueryToFilterUpstream(Resource, UpstreamModel) {
        const { created_after } = this.config;
        const upstreamTable = UpstreamModel.tableName;
        const baseQueryToFilterUpstream = await Resource.queryToFilterUpstream(upstreamTable);
        if (!created_after) {
            return baseQueryToFilterUpstream;
        }
        // Filter by created_at >= created_after
        const queryToFilterUpstream = {
            ...baseQueryToFilterUpstream
        };
        queryToFilterUpstream.where = {
            ...baseQueryToFilterUpstream?.where,
            created_at: {
                [_sequelize.Op.gte]: created_after
            }
        };
        return queryToFilterUpstream;
    }
    async countQueue() {
        let all = 0;
        for (const Resource of this.materialisableResources){
            const resourceTable = Resource.tableName;
            for (const UpstreamModel of Resource.UpstreamModels){
                const queryToFilterUpstream = await this.getQueryToFilterUpstream(Resource, UpstreamModel);
                const sql = await (0, _prepareQuery.prepareQuery)(UpstreamModel, {
                    ...queryToFilterUpstream,
                    attributes: [
                        'id'
                    ]
                });
                const [[{ total }]] = await Resource.sequelize.query(`
          WITH upstream AS (${sql.replace(/;$/, '')})
          SELECT COUNT(upstream.id) as total FROM upstream
          LEFT JOIN fhir."${resourceTable}" r ON r.upstream_id = upstream.id
          WHERE r.id IS NULL`);
                all += parseInt(total);
            }
        }
        return all;
    }
    async run() {
        for (const Resource of this.materialisableResources){
            const resourceTable = Resource.tableName;
            for (const UpstreamModel of Resource.UpstreamModels){
                const queryToFilterUpstream = await this.getQueryToFilterUpstream(Resource, UpstreamModel);
                const sql = await (0, _prepareQuery.prepareQuery)(UpstreamModel, {
                    ...queryToFilterUpstream,
                    attributes: [
                        'id'
                    ]
                });
                const [[{ total }]] = await Resource.sequelize.query(`
          WITH upstream AS (${sql.replace(/;$/, '')})
          SELECT COUNT(upstream.id) as total FROM upstream
          LEFT JOIN fhir."${resourceTable}" r ON r.upstream_id = upstream.id
          WHERE r.id IS NULL`);
                if (total === 0) {
                    this.log.debug('No missing resources to refresh', {
                        resource: Resource.fhirName
                    });
                    continue;
                }
                this.log.info('Submitting jobs to refresh missing resources', {
                    total,
                    resource: Resource.fhirName,
                    upstream: UpstreamModel.tableName
                });
                await Resource.sequelize.query(`
          WITH upstream AS (${sql.replace(/;$/, '')})
          INSERT INTO fhir.jobs (topic, payload, priority)
          SELECT
            $topic::text as topic,
            json_build_object(
              'resource', $resource::text,
              'upstreamId', upstream.id
            ) as payload,
            $priority::int as priority
          FROM upstream
          LEFT JOIN fhir."${resourceTable}" r ON r.upstream_id = upstream.id
          WHERE r.id IS NULL`, {
                    bind: {
                        topic: _constants.JOB_TOPICS.FHIR.REFRESH.FROM_UPSTREAM,
                        resource: Resource.fhirName,
                        priority: _constants.JOB_PRIORITIES.LOW
                    }
                });
            }
        }
    }
    constructor(context, overrideConfig = null){
        const conf = {
            ..._config.default.schedules.fhirMissingResources,
            ...overrideConfig
        };
        const { schedule, jitterTime, enabled } = conf;
        super(schedule, _logging.log.child({
            task: 'FhirMissingResources'
        }), jitterTime, enabled);
        this.config = conf;
        this.context = context;
        this.materialisableResources = (0, _resources.resourcesThatCanDo)(this.context.store.models, _constants.FHIR_INTERACTIONS.INTERNAL.MATERIALISE);
    }
};

//# sourceMappingURL=FhirMissingResources.js.map