"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.parseModuleRef = parseModuleRef;
exports.isPackageModuleRef = isPackageModuleRef;
exports.isFileModuleRef = isFileModuleRef;
exports.getPackageModuleRefFromModulePath = getPackageModuleRefFromModulePath;
const path_1 = __importDefault(require("path"));
const package_1 = require("./package");
/**
 * Parse a module reference into either a package module reference or a file
 * module reference.
 *
 * @param moduleRef Module reference
 * @return Module components
 */
function parseModuleRef(r) {
    if (r.startsWith(".") || path_1.default.isAbsolute(r)) {
        return {
            path: r,
        };
    }
    const ref = (0, package_1.parsePackageRef)(r);
    const indexPath = ref.name.indexOf("/");
    if (indexPath >= 0) {
        const p = ref.name.substring(indexPath + 1);
        if (p) {
            ref.path = p;
        }
        ref.name = ref.name.substring(0, indexPath);
    }
    return ref;
}
/**
 * Is the module reference a package module reference?
 *
 * @param r Module reference
 * @returns `true` if the module reference is based on a package
 */
function isPackageModuleRef(r) {
    return "name" in r;
}
/**
 * Is the module reference relative to a file location?
 *
 * @param r Module reference
 * @returns `true` if the module reference is based on a file location
 */
function isFileModuleRef(r) {
    return !isPackageModuleRef(r);
}
/**
 * Convert a module path to a package module reference.
 *
 * Find the module's `package.json` file, read the package name/scope, and
 * assemble that along with the module's path into a complete reference.
 *
 * @param modulePath Path to the module. Must be within a package.
 * @returns A package module reference, or `undefined` on failure
 */
function getPackageModuleRefFromModulePath(modulePath) {
    const pkgDir = (0, package_1.findPackageDir)(modulePath);
    if (pkgDir) {
        const manifest = (0, package_1.readPackage)(pkgDir);
        const ref = (0, package_1.parsePackageRef)(manifest.name);
        ref.path = path_1.default.relative(pkgDir, modulePath);
        return ref;
    }
    return undefined;
}
//# sourceMappingURL=module.js.map