"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "patientPortalMiddleware", {
    enumerable: true,
    get: function() {
        return patientPortalMiddleware;
    }
});
const _api = require("@opentelemetry/api");
const _expressasynchandler = /*#__PURE__*/ _interop_require_default(require("express-async-handler"));
const _config = /*#__PURE__*/ _interop_require_default(require("config"));
const _auth = require("@tamanu/constants/auth");
const _servers = require("@tamanu/constants/servers");
const _logging = require("@tamanu/shared/services/logging");
const _errors = require("@tamanu/errors");
const _createSessionIdentifier = require("@tamanu/shared/audit/createSessionIdentifier");
const _audit = require("@tamanu/database/utils/audit");
const _serverInfo = require("../../serverInfo");
const _utils = require("../../auth/utils");
function _interop_require_default(obj) {
    return obj && obj.__esModule ? obj : {
        default: obj
    };
}
const patientPortalMiddleware = ({ secret })=>(0, _expressasynchandler.default)(async (req, res, next)=>{
        const { store, headers } = req;
        const { canonicalHostName } = _config.default;
        const { authorization } = headers;
        if (!authorization) {
            throw new _errors.BadAuthenticationError('No authorization header provided');
        }
        const [bearer, token] = authorization.split(/\s/);
        const sessionId = (0, _createSessionIdentifier.createSessionIdentifier)(token);
        if (bearer.toLowerCase() !== 'bearer') {
            throw new _errors.BadAuthenticationError('Only Bearer token is supported');
        }
        let contents = null;
        try {
            contents = await (0, _utils.verifyToken)(token, secret, {
                issuer: canonicalHostName,
                audience: _auth.JWT_TOKEN_TYPES.PATIENT_PORTAL_ACCESS
            });
        } catch (e) {
            _logging.log.debug('Patient portal auth error: Invalid token', {
                error: e.message
            });
            res.status(401).send({
                error: {
                    message: 'Invalid token'
                }
            });
            return;
        }
        const { portalUserId } = contents.payload;
        const portalUser = await store.models.PortalUser.findByPk(portalUserId);
        if (!portalUser) {
            throw new _errors.BadAuthenticationError(`Portal user specified in token (${portalUserId}) does not exist`);
        }
        const patient = await portalUser.getPatient({
            include: [
                'additionalData'
            ]
        });
        if (!patient) {
            throw new _errors.BadAuthenticationError(`Portal user specified in token (${portalUserId}) does not have a patient`);
        }
        // Transform additionalData from array to object
        if (patient.additionalData && Array.isArray(patient.additionalData)) {
            patient.setDataValue('additionalData', patient.additionalData[0] || null);
        }
        /* eslint-disable require-atomic-updates */ req.portalUser = portalUser;
        req.patient = patient;
        req.sessionId = sessionId;
        /* eslint-enable require-atomic-updates */ const isAuditEnabled = await req.settings.get('audit.accesses.enabled');
        // Attach auditing helper similar to standard user middleware
        // eslint-disable-next-line require-atomic-updates
        req.audit = (0, _audit.initAuditActions)(req, {
            enabled: isAuditEnabled,
            userId: _auth.SYSTEM_USER_UUID,
            version: _serverInfo.version,
            backEndContext: {
                serverType: _servers.SERVER_TYPES.CENTRAL,
                isPatientPortal: true
            }
        });
        /**
     * Stub out permission checks to always return true. Permission checks are not needed or supported
     * in patient portal but these methods are called in shared code such as suggesters
     */ req.checkPermission = ()=>true;
        req.ability = {
            can: ()=>true
        };
        const spanAttributes = {
            'app.patient.id': patient.id
        };
        // eslint-disable-next-line no-unused-expressions
        _api.trace.getActiveSpan()?.setAttributes(spanAttributes);
        _api.context.with(_api.propagation.setBaggage(_api.context.active(), _api.propagation.createBaggage(spanAttributes)), ()=>next());
    });

//# sourceMappingURL=patientPortalMiddleware.js.map