"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
function _export(target, all) {
    for(var name in all)Object.defineProperty(target, name, {
        enumerable: true,
        get: all[name]
    });
}
_export(exports, {
    saveCreates: function() {
        return saveCreates;
    },
    saveDeletes: function() {
        return saveDeletes;
    },
    saveRestores: function() {
        return saveRestores;
    },
    saveUpdates: function() {
        return saveUpdates;
    }
});
const _sequelize = require("sequelize");
const _config = /*#__PURE__*/ _interop_require_default(require("config"));
const _tinyasyncpool = /*#__PURE__*/ _interop_require_default(require("tiny-async-pool"));
const _mergeRecord = require("./mergeRecord");
function _interop_require_default(obj) {
    return obj && obj.__esModule ? obj : {
        default: obj
    };
}
const persistUpdateWorkerPoolSize = _config.default.sync.persistUpdateWorkerPoolSize;
const saveCreates = async (model, records)=>{
    // can end up with duplicate create records, e.g. if syncAllLabRequests is turned on, an
    // encounter may turn up twice, once because it is for a marked-for-sync patient, and once more
    // because it has a lab request attached
    const deduplicated = [];
    const idsAdded = new Set();
    const idsForSoftDeleted = records.filter((row)=>row.isDeleted).map((row)=>row.id);
    for (const record of records){
        const data = {
            ...record
        };
        delete data.isDeleted;
        if (!idsAdded.has(data.id)) {
            deduplicated.push(data);
            idsAdded.add(data.id);
        }
    }
    await model.bulkCreate(deduplicated);
    // To create soft deleted records, we need to first create them, then destroy them
    if (idsForSoftDeleted.length > 0) {
        await model.destroy({
            where: {
                id: {
                    [_sequelize.Op.in]: idsForSoftDeleted
                }
            }
        });
    }
};
const saveUpdates = async (model, incomingRecords, idToExistingRecord, isCentralServer)=>{
    const recordsToSave = isCentralServer ? incomingRecords.map((incoming)=>{
        const existing = idToExistingRecord[incoming.id];
        return (0, _mergeRecord.mergeRecord)(existing, incoming);
    }) : incomingRecords;
    await (0, _tinyasyncpool.default)(persistUpdateWorkerPoolSize, recordsToSave, async (r)=>model.update(r, {
            where: {
                id: r.id
            },
            paranoid: false
        }));
};
const saveDeletes = async (model, recordsForDelete)=>{
    if (recordsForDelete.length === 0) return;
    await model.destroy({
        where: {
            id: {
                [_sequelize.Op.in]: recordsForDelete.map((r)=>r.id)
            }
        }
    });
};
const saveRestores = async (model, recordsForRestore)=>{
    if (recordsForRestore.length === 0) return;
    await model.restore({
        where: {
            id: {
                [_sequelize.Op.in]: recordsForRestore.map((r)=>r.id)
            }
        }
    });
};

//# sourceMappingURL=saveChanges.js.map