"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
function _export(target, all) {
    for(var name in all)Object.defineProperty(target, name, {
        enumerable: true,
        get: all[name]
    });
}
_export(exports, {
    PortalOneTimeTokenService: function() {
        return PortalOneTimeTokenService;
    },
    hashPortalToken: function() {
        return hashPortalToken;
    }
});
const _datefns = require("date-fns");
const _crypto = require("crypto");
const _bcrypt = /*#__PURE__*/ _interop_require_default(require("bcrypt"));
const _config = /*#__PURE__*/ _interop_require_default(require("config"));
const _constants = require("@tamanu/constants");
const _errors = require("@tamanu/errors");
function _interop_require_default(obj) {
    return obj && obj.__esModule ? obj : {
        default: obj
    };
}
const DEFAULT_SALT_ROUNDS = 10;
function randomSixDigitCode() {
    // returns a zero-padded 6 digit string using crypto.randomInt for better security
    return Array.from({
        length: 6
    }, ()=>(0, _crypto.randomInt)(0, 9)).join('');
}
function randomRegisterCode() {
    // secure random code for magic links - 32 hex chars
    return (0, _crypto.randomBytes)(16).toString('hex');
}
async function hashPortalToken(token) {
    return _bcrypt.default.hash(token, DEFAULT_SALT_ROUNDS);
}
let PortalOneTimeTokenService = class PortalOneTimeTokenService {
    async createLoginToken(portalUserId) {
        const { PortalOneTimeToken } = this.models;
        const token = randomSixDigitCode();
        const hashedToken = await hashPortalToken(token);
        const expiresAt = (0, _datefns.addMinutes)(new Date(), _config.default.patientPortal.loginTokenDurationMinutes);
        // Overwrite existing login tokens for this user
        await PortalOneTimeToken.destroy({
            where: {
                portalUserId,
                type: _constants.PORTAL_ONE_TIME_TOKEN_TYPES.LOGIN
            },
            force: true
        });
        const record = await PortalOneTimeToken.create({
            portalUserId,
            type: _constants.PORTAL_ONE_TIME_TOKEN_TYPES.LOGIN,
            token: hashedToken,
            expiresAt
        });
        return {
            token,
            expiresAt: record.expiresAt
        };
    }
    async createRegisterToken(portalUserId) {
        const { PortalOneTimeToken } = this.models;
        const token = randomRegisterCode();
        const hashedToken = await hashPortalToken(token);
        const expiresAt = (0, _datefns.addMinutes)(new Date(), _config.default.patientPortal.registerTokenDurationMinutes);
        // Overwrite existing register tokens for this user
        await PortalOneTimeToken.destroy({
            where: {
                portalUserId,
                type: _constants.PORTAL_ONE_TIME_TOKEN_TYPES.REGISTER
            },
            force: true
        });
        const record = await PortalOneTimeToken.create({
            portalUserId,
            type: _constants.PORTAL_ONE_TIME_TOKEN_TYPES.REGISTER,
            token: hashedToken,
            expiresAt
        });
        return {
            token,
            expiresAt: record.expiresAt
        };
    }
    async verifyAndConsume({ portalUserId, token, type = _constants.PORTAL_ONE_TIME_TOKEN_TYPES.LOGIN }) {
        const { PortalOneTimeToken, PortalUser } = this.models;
        const portalUser = await PortalUser.findByPk(portalUserId);
        if (!portalUser) {
            throw new _errors.InvalidCredentialError('Invalid verification code');
        }
        const tokenRecord = await PortalOneTimeToken.findOne({
            where: {
                portalUserId,
                type
            },
            order: [
                [
                    'expiresAt',
                    'DESC'
                ]
            ]
        });
        if (!tokenRecord) {
            throw new _errors.InvalidCredentialError('Invalid verification code');
        }
        if (tokenRecord.isExpired()) {
            throw new _errors.InvalidTokenError('Verification code has expired');
        }
        const isVerified = await _bcrypt.default.compare(token, tokenRecord.token);
        if (!isVerified) {
            throw new _errors.InvalidCredentialError('Invalid verification code');
        }
        await tokenRecord.destroy({
            where: {
                id: tokenRecord.id
            },
            force: true
        });
        return portalUser;
    }
    constructor(models){
        this.models = models;
    }
};

//# sourceMappingURL=PortalOneTimeTokenService.js.map