"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
function _export(target, all) {
    for(var name in all)Object.defineProperty(target, name, {
        enumerable: true,
        get: all[name]
    });
}
_export(exports, {
    ACTIVE_TEXT: function() {
        return ACTIVE_TEXT;
    },
    DEFAULT_USER_EMAIL: function() {
        return DEFAULT_USER_EMAIL;
    },
    OVERWRITING_TEXT: function() {
        return OVERWRITING_TEXT;
    },
    createVersion: function() {
        return createVersion;
    },
    formatUpdatedAt: function() {
        return formatUpdatedAt;
    },
    getVersionError: function() {
        return getVersionError;
    },
    listVersions: function() {
        return listVersions;
    }
});
const _logging = require("@tamanu/shared/services/logging");
const _clitable3 = /*#__PURE__*/ _interop_require_default(require("cli-table3"));
const _constants = require("@tamanu/constants");
const _datefns = require("date-fns");
const _utils = require("./utils");
function _interop_require_default(obj) {
    return obj && obj.__esModule ? obj : {
        default: obj
    };
}
const DEFAULT_USER_EMAIL = 'admin@tamanu.io';
const colorise = (colorCode, text)=>`\x1b[${colorCode}m${text}\x1b[0m`;
const ACTIVE_TEXT = colorise('32', 'active');
const OVERWRITING_TEXT = colorise('1m', 'overwriting with new data');
const formatUpdatedAt = (date)=>(0, _datefns.format)(date, 'P p');
const getVersionError = ({ versionNumber })=>new Error(`Version ${versionNumber} does not exist, remove versionNumber from JSON and try again to auto increment`);
async function createVersion(versionData, definition, versions, store, verbose) {
    const { ReportDefinitionVersion } = store.models;
    _logging.log.info('Analyzing query');
    await (0, _utils.verifyQuery)(versionData.query, versionData.queryOptions?.parameters, store, verbose);
    _logging.log.info('Query is valid');
    let { id, versionNumber } = versionData;
    if (Number.isInteger(versionData.versionNumber)) {
        const existingVersion = versions.find((v)=>v.versionNumber === versionData.versionNumber);
        if (!existingVersion) {
            throw getVersionError(versionData);
        }
        _logging.log.warn(`Version ${versionData.versionNumber} already exists, ${OVERWRITING_TEXT}`);
        id = existingVersion.id;
    } else {
        const latestVersion = (0, _utils.getLatestVersion)(versions);
        versionNumber = (latestVersion?.versionNumber || 0) + 1;
        _logging.log.info(`Auto incrementing versionNumber to ${versionNumber}`);
    }
    let { userId } = versionData;
    if (!versionData.userId) {
        _logging.log.warn(`User id not specified`);
        const user = await store.models.User.findOne({
            where: {
                email: DEFAULT_USER_EMAIL,
                visibilityStatus: _constants.VISIBILITY_STATUSES.CURRENT
            }
        });
        _logging.log.info(`Using default user ${DEFAULT_USER_EMAIL}`);
        userId = user.id;
    }
    const [version] = await ReportDefinitionVersion.upsert({
        ...versionData,
        id,
        versionNumber,
        userId,
        reportDefinitionId: definition.id
    });
    const created = !versionData.id;
    const isActive = version.status === _constants.REPORT_STATUSES.PUBLISHED && (created || (0, _utils.getLatestVersion)(versions).versionNumber === version.versionNumber);
    _logging.log.info('Imported new report version', {
        action: created ? 'createdNew' : 'updated',
        active: isActive,
        userId,
        versionNumber: version.versionNumber,
        definitionName: definition.name,
        definitionId: definition.id
    });
}
async function listVersions(definition, versions) {
    if (!versions.length) {
        _logging.log.info(`No versions found for report definition ${definition.name}`);
        return;
    }
    const activeVersion = (0, _utils.getLatestVersion)(versions, _constants.REPORT_STATUSES.PUBLISHED);
    const table = new _clitable3.default({
        head: [
            'Version',
            'Status',
            'Updated'
        ]
    });
    versions.forEach(({ versionNumber, status, updatedAt })=>{
        const isActive = activeVersion?.versionNumber === versionNumber;
        table.push([
            versionNumber,
            `${status}${isActive ? ` ${ACTIVE_TEXT}` : ''}`,
            formatUpdatedAt(updatedAt)
        ]);
    });
    _logging.log.info(`Listing versions for report definition ${definition.name}\n${table.toString()}`);
}

//# sourceMappingURL=actions.js.map