"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
function _export(target, all) {
    for(var name in all)Object.defineProperty(target, name, {
        enumerable: true,
        get: all[name]
    });
}
_export(exports, {
    closeAllDatabases: function() {
        return closeAllDatabases;
    },
    databaseCollection: function() {
        return databaseCollection;
    },
    initDatabase: function() {
        return initDatabase;
    },
    openDatabase: function() {
        return openDatabase;
    }
});
const _async_hooks = require("async_hooks");
const _sequelize = require("sequelize");
const _pg = /*#__PURE__*/ _interop_require_default(require("pg"));
const _util = /*#__PURE__*/ _interop_require_default(require("util"));
const _constants = require("@tamanu/constants");
const _logging = require("@tamanu/shared/services/logging");
const _context = require("@tamanu/shared/services/logging/context");
const _migrations = require("./migrations");
const _models = /*#__PURE__*/ _interop_require_wildcard(require("../models"));
const _createDateTypes = require("./createDateTypes");
const _pgComposite = require("../utils/pgComposite");
const _utils = require("../utils");
function _interop_require_default(obj) {
    return obj && obj.__esModule ? obj : {
        default: obj
    };
}
function _getRequireWildcardCache(nodeInterop) {
    if (typeof WeakMap !== "function") return null;
    var cacheBabelInterop = new WeakMap();
    var cacheNodeInterop = new WeakMap();
    return (_getRequireWildcardCache = function(nodeInterop) {
        return nodeInterop ? cacheNodeInterop : cacheBabelInterop;
    })(nodeInterop);
}
function _interop_require_wildcard(obj, nodeInterop) {
    if (!nodeInterop && obj && obj.__esModule) {
        return obj;
    }
    if (obj === null || typeof obj !== "object" && typeof obj !== "function") {
        return {
            default: obj
        };
    }
    var cache = _getRequireWildcardCache(nodeInterop);
    if (cache && cache.has(obj)) {
        return cache.get(obj);
    }
    var newObj = {
        __proto__: null
    };
    var hasPropertyDescriptor = Object.defineProperty && Object.getOwnPropertyDescriptor;
    for(var key in obj){
        if (key !== "default" && Object.prototype.hasOwnProperty.call(obj, key)) {
            var desc = hasPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : null;
            if (desc && (desc.get || desc.set)) {
                Object.defineProperty(newObj, key, desc);
            } else {
                newObj[key] = obj[key];
            }
        }
    }
    newObj.default = obj;
    if (cache) {
        cache.set(obj, newObj);
    }
    return newObj;
}
(0, _createDateTypes.createDateTypes)();
// This allows us to use transaction callbacks without manually managing a transaction handle
// https://sequelize.org/master/manual/transactions.html#automatically-pass-transactions-to-all-queries
// done once for all sequelize objects. Instead of cls-hooked we use the built-in AsyncLocalStorage.
// NOTE: Do not use this storage for anything, sequelize manages it and may clear it unexpectedly
const sequelizeTransactionAsyncLocalStorage = new _async_hooks.AsyncLocalStorage();
// eslint-disable-next-line react-hooks/rules-of-hooks
_sequelize.Sequelize.useCLS({
    bind: ()=>{},
    get: (id)=>sequelizeTransactionAsyncLocalStorage.getStore()?.get(id),
    set: (id, value)=>sequelizeTransactionAsyncLocalStorage.getStore()?.set(id, value),
    run: (callback)=>sequelizeTransactionAsyncLocalStorage.run(new Map(), callback)
});
// When a transaction is started, Sequelize wraps all calls in a `run()` call on the clsAsyncLocalStorage,
// so we know we're in a transaction if the value is not empty
const isInsideTransaction = ()=>Boolean(sequelizeTransactionAsyncLocalStorage.getStore());
// Once Sequelize has set up the transaction (ie. calling 'START TRANSACTION' and 'SET ISOLATION LEVEL ...') it will add a 'transaction' object to the store
const isTransactionReady = ()=>Boolean(sequelizeTransactionAsyncLocalStorage.getStore()?.get('transaction'));
// this is dangerous and should only be used in test mode
const unsafeRecreatePgDb = async ({ name, username, password, host, port })=>{
    const client = new _pg.default.Client({
        user: username,
        password,
        host,
        port,
        // 'postgres' is the default database that's automatically
        // created on new installs - we just need something to connect
        // to, and it doesn't matter what the schema is!
        database: 'postgres'
    });
    try {
        await client.connect();
        await client.query(`DROP DATABASE IF EXISTS "${name}"`);
        await client.query(`CREATE DATABASE "${name}"`);
    } catch (e) {
        _logging.log.error('Failed to drop database. Note that every createTestContext() must have a corresponding ctx.close()!');
        throw e;
    } finally{
        await client.end();
    }
};
async function connectToDatabase(dbOptions) {
    // connect to database
    const { username, password, testMode = false, host = null, port = null, verbose = false, pool, alwaysCreateConnection = true, loggingOverride = null, disableChangesAudit = false, recreateDatabase = false } = dbOptions;
    let { name } = dbOptions;
    // configure one test db per jest worker
    const workerId = process.env.JEST_WORKER_ID;
    if (testMode && (workerId || recreateDatabase)) {
        if (workerId) {
            name = `${name}-${workerId}`;
        }
        if (alwaysCreateConnection) {
            await unsafeRecreatePgDb({
                ...dbOptions,
                name
            });
        }
    }
    _logging.log.info('databaseConnection', {
        username,
        host,
        port,
        name
    });
    let logging;
    if (loggingOverride) {
        logging = loggingOverride;
    } else if (verbose) {
        logging = (query, obj)=>_logging.log.debug('databaseQuery', {
                query: _util.default.inspect(query),
                binding: _util.default.inspect(obj.bind || [], {
                    breakLength: Infinity
                })
            });
    } else {
        logging = null;
    }
    const options = {
        dialect: 'postgres',
        dialectOptions: {
            application_name: (0, _context.serviceName)((0, _context.serviceContext)()) ?? 'tamanu'
        }
    };
    const sequelize = new _sequelize.Sequelize(name, username, password, {
        ...options,
        host,
        port,
        logging,
        pool
    });
    sequelize.setSessionVar = (key, value)=>sequelize.query(`SELECT public.set_session_config($key, $value)`, {
            bind: {
                key,
                value
            }
        });
    sequelize.setTransactionVar = (key, value)=>sequelize.query(`SELECT public.set_session_config($key, $value, true)`, {
            bind: {
                key,
                value
            }
        });
    if (!disableChangesAudit) {
        let QueryWithAuditConfig = class QueryWithAuditConfig extends sequelize.dialect.Query {
            async run(sql, options) {
                const userid = (0, _utils.getAuditUserId)();
                const isInsideATransaction = isInsideTransaction();
                const isThisTransactionReady = isTransactionReady();
                // Set audit userid so that any changes in this query are recorded against it
                // If in a transaction, just use a transaction scoped variable to avoid needing to clear it
                if (userid) {
                    if (isInsideATransaction && !isThisTransactionReady) {
                    // This may be the 'START TRANSACTION' or 'SET ISOLATION LEVEL ...' queries being run by sequelize.
                    // We don't want to run any queries until the transaction has been set up. So do nothing.
                    } else {
                        await super.run('SELECT public.set_session_config($1, $2, $3)', [
                            _constants.AUDIT_USERID_KEY,
                            userid,
                            isInsideATransaction
                        ]);
                    }
                }
                try {
                    return await super.run(sql, options);
                } catch (error) {
                    _logging.log.error(error);
                    throw error;
                } finally{
                    // Clear audit userid so that system user changes aren't unintentionally recorded against it
                    if (userid && !isInsideATransaction) {
                        await super.run('SELECT public.set_session_config($1, $2)', [
                            _constants.AUDIT_USERID_KEY,
                            _constants.SYSTEM_USER_UUID
                        ]);
                    }
                }
            }
        };
        sequelize.dialect.Query = QueryWithAuditConfig;
    }
    (0, _pgComposite.setupQuote)(sequelize);
    await sequelize.authenticate();
    if (!testMode) {
        // in test mode the context is closed manually, and we spin up lots of database
        // connections, so this is just holding onto the sequelize instance in a callback
        // that never gets called.
        process.once('SIGTERM', ()=>{
            _logging.log.info('Received SIGTERM, closing sequelize');
            sequelize.close();
        });
    }
    return sequelize;
}
async function initDatabase(dbOptions) {
    // connect to database
    const { makeEveryModelParanoid = true, saltRounds = null, alwaysCreateConnection = true, primaryKeyDefault = _sequelize.Sequelize.fn('gen_random_uuid'), hackToSkipEncounterValidation = false } = dbOptions;
    const sequelize = await connectToDatabase(dbOptions);
    if (!alwaysCreateConnection) {
        return {
            sequelize
        };
    }
    // set configuration variables for individual models
    _models.User.SALT_ROUNDS = saltRounds;
    // attach migration function to the sequelize object - leaving the responsibility
    // of calling it to the implementing server (this allows for skipping migrations
    // in favour of calling sequelize.sync() during test mode)
    // eslint-disable-next-line require-atomic-updates
    sequelize.migrate = async (direction)=>{
        await (0, _migrations.migrate)(_logging.log, sequelize, direction);
    };
    sequelize.assertUpToDate = async (options)=>(0, _migrations.assertUpToDate)(_logging.log, sequelize, options);
    // init all models
    const modelClasses = Object.values(_models);
    const primaryKey = {
        type: _sequelize.Sequelize.STRING,
        defaultValue: primaryKeyDefault,
        allowNull: false,
        primaryKey: true
    };
    _logging.log.info('registeringModels', {
        count: modelClasses.length
    });
    modelClasses.forEach((modelClass)=>{
        if ('initModel' in modelClass) {
            modelClass.initModel({
                underscored: true,
                primaryKey,
                sequelize,
                paranoid: makeEveryModelParanoid,
                hackToSkipEncounterValidation
            }, _models);
        } else {
            throw new Error(`Model ${modelClass.name} has no initModel()`);
        }
    });
    modelClasses.forEach((modelClass)=>{
        if (modelClass.initRelations) {
            modelClass.initRelations(_models);
        }
    });
    modelClasses.forEach((modelClass)=>{
        if (modelClass.syncDirection === _constants.SYNC_DIRECTIONS.DO_NOT_SYNC && modelClass.usesPublicSchema && !_migrations.NON_SYNCING_TABLES.includes(`public.${modelClass.tableName}`)) {
            throw new Error(`Any table that does not sync should be added to the "NON_SYNCING_TABLES" list. Please check ${modelClass.tableName}`);
        }
    });
    // add isInsideTransaction and isTransactionReady helpers to avoid exposing the asynclocalstorage
    sequelize.isInsideTransaction = isInsideTransaction;
    sequelize.isTransactionReady = isTransactionReady;
    return {
        sequelize,
        models: _models
    };
}
const databaseCollection = new Map();
async function openDatabase(key, dbOptions) {
    if (databaseCollection.has(key)) {
        return databaseCollection.get(key);
    }
    const store = await initDatabase(dbOptions);
    if (databaseCollection.has(key)) {
        throw new Error(`race condition! openDatabase() called for the same key=${key} in parallel`);
    }
    databaseCollection.set(key, store);
    return store;
}
async function closeAllDatabases() {
    for (const [key, connection] of databaseCollection.entries()){
        databaseCollection.delete(key);
        await connection.sequelize.close();
    }
}

//# sourceMappingURL=database.js.map