"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "performTimeZoneChecks", {
    enumerable: true,
    get: function() {
        return performTimeZoneChecks;
    }
});
const _sequelize = require("sequelize");
const _logging = require("../services/logging");
function getSystemTimeZone() {
    return Intl.DateTimeFormat().resolvedOptions().timeZone;
}
function getConfigTimeZone(config) {
    return config.countryTimeZone;
}
async function getDatabaseTimeZone(sequelize) {
    const rows = await sequelize.query("SELECT setting FROM pg_settings WHERE name ILIKE 'timezone'", {
        type: _sequelize.QueryTypes.SELECT
    });
    return rows[0].setting;
}
// Try to grab current remote time zone
// otherwise ignore that one
async function getRemoteTimeZone(remote) {
    try {
        const health = await remote.fetch('health', {
            timeout: 2000,
            backoff: {
                maxAttempts: 1
            },
            preserveBackoffForAuthAttempt: true
        });
        const { countryTimeZone } = health.config;
        return countryTimeZone;
    } catch (error) {
        _logging.log.warn('Unable to grab countryTimeZone from central server.');
    }
    return null;
}
async function performTimeZoneChecks({ config, sequelize, remote }) {
    const zones = {
        system: getSystemTimeZone(),
        config: getConfigTimeZone(config),
        database: await getDatabaseTimeZone(sequelize)
    };
    if (remote) {
        zones.remoteConfig = await getRemoteTimeZone(remote);
    }
    /*
  TODO: 
  When Sequelize connects to Postgres without an explicit timezone parameter, it causes 
  it to report its timezone slightly weirdly (as '<+00>-00' rather than a named TZ).
  But providing the timezone explicitly breaks some of our reports...! 
  So just log the timezones for now and we perform the more rigid check once
  we've sorted those issues out.
  */ _logging.log.info('Checking timezone consistency', zones);
/*
  const unique = new Set(Object.values(zones));
  if (unique.size > 1) {
    const errorText = `Detected mismatched time zones. Details: ${JSON.stringify(zones)}.`;
    if (config.allowMismatchedTimeZones) {
      log.warn(errorText);
    } else {
      throw new InvalidConfigError(
        `${errorText} Please ensure these are consistent, or set config.allowMismatchedTimeZones to true.`,
      );
    }
  }
  */ }

//# sourceMappingURL=timeZoneCheck.js.map