"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "FhirDiagnosticReport", {
    enumerable: true,
    get: function() {
        return FhirDiagnosticReport;
    }
});
const _sequelize = require("sequelize");
const _yup = /*#__PURE__*/ _interop_require_wildcard(require("yup"));
const _constants = require("@tamanu/constants");
const _dateTime = require("@tamanu/utils/dateTime");
const _errors = require("@tamanu/errors");
const _fhirTypes = require("@tamanu/shared/services/fhirTypes");
const _fhir = require("@tamanu/shared/utils/fhir");
const _Resource = require("./Resource");
function _define_property(obj, key, value) {
    if (key in obj) {
        Object.defineProperty(obj, key, {
            value: value,
            enumerable: true,
            configurable: true,
            writable: true
        });
    } else {
        obj[key] = value;
    }
    return obj;
}
function _getRequireWildcardCache(nodeInterop) {
    if (typeof WeakMap !== "function") return null;
    var cacheBabelInterop = new WeakMap();
    var cacheNodeInterop = new WeakMap();
    return (_getRequireWildcardCache = function(nodeInterop) {
        return nodeInterop ? cacheNodeInterop : cacheBabelInterop;
    })(nodeInterop);
}
function _interop_require_wildcard(obj, nodeInterop) {
    if (!nodeInterop && obj && obj.__esModule) {
        return obj;
    }
    if (obj === null || typeof obj !== "object" && typeof obj !== "function") {
        return {
            default: obj
        };
    }
    var cache = _getRequireWildcardCache(nodeInterop);
    if (cache && cache.has(obj)) {
        return cache.get(obj);
    }
    var newObj = {
        __proto__: null
    };
    var hasPropertyDescriptor = Object.defineProperty && Object.getOwnPropertyDescriptor;
    for(var key in obj){
        if (key !== "default" && Object.prototype.hasOwnProperty.call(obj, key)) {
            var desc = hasPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : null;
            if (desc && (desc.get || desc.set)) {
                Object.defineProperty(newObj, key, desc);
            } else {
                newObj[key] = obj[key];
            }
        }
    }
    newObj.default = obj;
    if (cache) {
        cache.set(obj, newObj);
    }
    return newObj;
}
let FhirDiagnosticReport = class FhirDiagnosticReport extends _Resource.FhirResource {
    static initModel(options, models) {
        super.initResource({
            basedOn: {
                type: _sequelize.DataTypes.JSONB,
                allowNull: false
            },
            status: {
                type: _sequelize.DataTypes.TEXT,
                allowNull: false
            },
            code: {
                type: _sequelize.DataTypes.JSONB,
                allowNull: false
            },
            presentedForm: {
                type: _sequelize.DataTypes.JSONB
            }
        }, options);
        this.UpstreamModels = [
            models.LabTest
        ];
        this.upstreams = [
            models.LabTest,
            models.LabRequest,
            models.LabTestType,
            models.LabRequestAttachment
        ];
    }
    static get INTAKE_SCHEMA() {
        return _yup.object({
            basedOn: _yup.array().of(_fhirTypes.FhirReference.asYup()).required(),
            status: _yup.string().required(),
            code: _fhirTypes.FhirCodeableConcept.asYup().required(),
            presentedForm: _yup.array().of(_yup.object({
                data: _yup.string().required(),
                title: _yup.string().required(),
                contentType: _yup.string().required()
            }))
        });
    }
    // This is beginning very modestly - can extend to handle full
    // results soon.
    async pushUpstream({ requesterId }) {
        const { FhirServiceRequest, LabRequest } = this.sequelize.models;
        if (!this.basedOn || !Array.isArray(this.basedOn)) {
            throw new _fhir.Invalid('DiagnosticReport requires basedOn to report results for ServiceRequest', {
                code: _constants.FHIR_ISSUE_TYPE.INVALID.VALUE
            });
        }
        const { type, reference } = this.basedOn[0];
        const ref = reference.split('/');
        if (type !== 'ServiceRequest' || ref.length < 2 || ref[0] !== 'ServiceRequest') {
            throw new _fhir.Invalid(`DiagnosticReport requires must be results for ServiceRequest'`, {
                code: _constants.FHIR_ISSUE_TYPE.INVALID.VALUE
            });
        }
        const serviceRequestFhirId = ref[1];
        const serviceRequest = await FhirServiceRequest.findOne({
            where: {
                id: serviceRequestFhirId
            }
        });
        if (!serviceRequest) {
            throw new _fhir.Invalid(`ServiceRequest '${serviceRequestFhirId}' does not exist in Tamanu`, {
                code: _constants.FHIR_ISSUE_TYPE.INVALID.VALUE
            });
        }
        if (!this.getLabRequestStatus()) {
            throw new _fhir.Invalid(`LabRequest status invalid`, {
                code: _constants.FHIR_ISSUE_TYPE.INVALID.VALUE
            });
        }
        const labRequest = await LabRequest.findByPk(serviceRequest.upstreamId);
        if (!labRequest) {
            throw new _fhir.Invalid(`No LabRequest with id: '${serviceRequest.upstreamId}', might be ImagingRequest id`);
        }
        await this.sequelize.transaction(async ()=>{
            const newStatus = this.getLabRequestStatus();
            if (this.shouldUpdateLabRequestStatus(labRequest, newStatus)) {
                labRequest.set({
                    status: newStatus
                });
                if (newStatus === _constants.LAB_REQUEST_STATUSES.PUBLISHED) {
                    labRequest.set({
                        publishedDate: (0, _dateTime.getCurrentDateTimeString)()
                    });
                }
                await labRequest.save();
                if (!requesterId) throw new _errors.InvalidOperationError('No user found for LabRequest status change.');
                await this.sequelize.models.LabRequestLog.create({
                    status: newStatus,
                    labRequestId: labRequest.id,
                    updatedById: requesterId
                });
            }
        });
        if (this.presentedForm) {
            await this.saveAttachment(labRequest);
        }
        return labRequest;
    }
    getLabRequestStatus() {
        if (this.status === _constants.FHIR_DIAGNOSTIC_REPORT_STATUS.REGISTERED) {
            return _constants.LAB_REQUEST_STATUSES.RESULTS_PENDING;
        }
        if (this.status === _constants.FHIR_DIAGNOSTIC_REPORT_STATUS.PARTIAL._) {
            if (this.presentedForm) {
                return _constants.LAB_REQUEST_STATUSES.INTERIM_RESULTS;
            } else {
                return _constants.LAB_REQUEST_STATUSES.RESULTS_PENDING;
            }
        }
        if (this.status === _constants.FHIR_DIAGNOSTIC_REPORT_STATUS.PARTIAL.PRELIMINARY) {
            if (this.presentedForm) {
                return _constants.LAB_REQUEST_STATUSES.INTERIM_RESULTS;
            } else {
                return _constants.LAB_REQUEST_STATUSES.TO_BE_VERIFIED;
            }
        }
        if (this.status === _constants.FHIR_DIAGNOSTIC_REPORT_STATUS.FINAL) {
            if (this.presentedForm) {
                return _constants.LAB_REQUEST_STATUSES.PUBLISHED;
            } else {
                return _constants.LAB_REQUEST_STATUSES.VERIFIED;
            }
        }
        if (this.status === _constants.FHIR_DIAGNOSTIC_REPORT_STATUS.CANCELLED) {
            return _constants.LAB_REQUEST_STATUSES.CANCELLED;
        }
        if (this.status === _constants.FHIR_DIAGNOSTIC_REPORT_STATUS.ENTERED_IN_ERROR) {
            return _constants.LAB_REQUEST_STATUSES.ENTERED_IN_ERROR;
        }
        if (this.status === _constants.FHIR_DIAGNOSTIC_REPORT_STATUS.AMENDED._) {
            return _constants.LAB_REQUEST_STATUSES.INVALIDATED;
        }
        if (this.status === _constants.FHIR_DIAGNOSTIC_REPORT_STATUS.AMENDED.APPENDED || this.status === _constants.FHIR_DIAGNOSTIC_REPORT_STATUS.AMENDED.CORRECTED) {
            // no workflow for these yet
            throw new _fhir.Invalid(`${this.status} workflow unsupported`);
        }
        throw new _fhir.Invalid(`'${this.status}' is an invalid ServiceRequest status`);
    }
    shouldUpdateLabRequestStatus(labRequest, newStatus) {
        if (!labRequest.status) {
            return false; // Don't update a status for a labRequest that doesn't support it
        }
        if (labRequest.status === newStatus) {
            return false; // No need to update if not changing the status
        }
        if (labRequest.status === _constants.LAB_REQUEST_STATUSES.PUBLISHED) {
            // Once a labRequest has been published, we can only change the status to INVALIDATED. Ignore all other status changes
            if (newStatus === _constants.LAB_REQUEST_STATUSES.INVALIDATED) {
                return true;
            }
            return false;
        }
        return true;
    }
    async saveAttachment(labRequest) {
        if (!Array.isArray(this.presentedForm) || this.presentedForm.length > 1) {
            throw new _fhir.Invalid('presentedForm must be an array of length 1');
        }
        const form = this.presentedForm[0];
        if (!Object.values(_constants.SUPPORTED_CONTENT_TYPES).includes(form.contentType)) {
            throw new _fhir.Invalid(`presentedForm must be one of the supported values: ${Object.values(_constants.SUPPORTED_CONTENT_TYPES)}`);
        }
        const { Attachment, LabRequestAttachment } = this.sequelize.models;
        const { data, type } = Attachment.sanitizeForDatabase({
            data: form.data,
            type: form.contentType
        });
        const attachment = await Attachment.create({
            data,
            type
        });
        const lastAttachment = await labRequest.getLatestAttachment();
        const labRequestAttachment = await LabRequestAttachment.create({
            attachmentId: attachment.id,
            title: form.title,
            labRequestId: labRequest.id,
            isVisible: true
        });
        if (lastAttachment) {
            lastAttachment.set({
                replacedById: labRequestAttachment.id
            });
            await lastAttachment.save();
        }
    }
};
_define_property(FhirDiagnosticReport, "CAN_DO", new Set([
    _constants.FHIR_INTERACTIONS.TYPE.CREATE
]));

//# sourceMappingURL=FhirDiagnosticReport.js.map